within ScalableTestSuite.Electrical;
package DistributionSystemAC
  import SI = Modelica.SIunits;

  package Models
    package Internals
      operator record ComplexVoltage = SI.ComplexVoltage(re(nominal = 1e3), im(nominal = 1e3));
      operator record ComplexCurrent = SI.ComplexCurrent(re(nominal = 1e3), im(nominal = 1e3));
      operator record ComplexPower = SI.ComplexPower(re(nominal = 1e4), im(nominal = 1e4));

      connector Pin "Pin of an electrical component"
        ComplexVoltage v "Potential at the pin";
        flow ComplexCurrent i "Current flowing into the pin";
        annotation(defaultComponentName = "pin", Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-100, 100}, {100, -100}}, lineColor = {0, 0, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid)}), Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-40, 40}, {40, -40}}, lineColor = {0, 0, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Text(extent = {{-160, 110}, {40, 50}}, lineColor = {0, 0, 255}, textString = "%name")}));
      end Pin;

      connector PositivePin "Positive pin of an electric component"
        ComplexVoltage v "Potential at the pin";
        flow ComplexCurrent i "Current flowing into the pin";
        annotation(defaultComponentName = "p", Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-100, 100}, {100, -100}}, lineColor = {0, 0, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid)}), Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-40, 40}, {40, -40}}, lineColor = {0, 0, 255}, fillColor = {0, 0, 255}, fillPattern = FillPattern.Solid), Text(extent = {{-160, 110}, {40, 50}}, lineColor = {0, 0, 255}, textString = "%name")}));
      end PositivePin;

      connector NegativePin "Negative pin of an electric component"
        ComplexVoltage v "Potential at the pin";
        flow ComplexCurrent i "Current flowing into the pin";
        annotation(defaultComponentName = "n", Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-100, 100}, {100, -100}}, lineColor = {0, 0, 255}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Solid)}), Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100, -100}, {100, 100}}), graphics = {Rectangle(extent = {{-40, 40}, {40, -40}}, lineColor = {0, 0, 255}, fillColor = {255, 255, 255}, fillPattern = FillPattern.Solid), Text(extent = {{-40, 110}, {160, 50}}, textString = "%name", lineColor = {0, 0, 255})}));
      end NegativePin;

      model Ground "Ground model"
        ScalableTestSuite.Electrical.DistributionSystemAC.Models.Internals.Pin p annotation(Placement(visible = true, transformation(origin = {0, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {0, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
      equation
        p.v = Complex(0, 0);
        annotation(uses(ScalableTestSuite(version = "1.7.1")), Icon(graphics = {Line(origin = {0, -25}, points = {{0, 15}, {0, -15}}), Line(origin = {1, -40}, points = {{-59, 0}, {59, 0}}), Line(origin = {0, -60}, points = {{-40, 0}, {40, 0}}), Line(origin = {0, -79}, points = {{-20, -1}, {20, -1}}), Line(origin = {-1, -100}, points = {{-1, 0}, {3, 0}})}));
      end Ground;

      partial model OnePort "Generic one-port model with complex voltage and current"
        PositivePin p annotation(Placement(visible = true, transformation(origin = {-100, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {-100, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
        NegativePin n annotation(Placement(visible = true, transformation(origin = {100, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {100, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
        SI.ComplexVoltage v "Voltage across component";
        SI.ComplexCurrent i "Current through component";
      equation
        v = p.v - n.v;
        p.i + n.i = Complex(0, 0);
        i = p.i;
        annotation(Icon(coordinateSystem(initialScale = 0.1)));
      end OnePort;

      model VoltageSource "Fixed Real voltage source"
        extends Internals.OnePort;
        parameter SI.Voltage V "Fixed (real) source voltage";
      equation
        v = Complex(V, 0) "Enforce prescribed voltage";
        annotation(Icon(graphics = {Line(origin = {1, -1}, points = {{-91, 1}, {89, 1}}), Ellipse(origin = {0, 1}, extent = {{-40, 39}, {40, -41}}, endAngle = 360)}, coordinateSystem(initialScale = 0.1)));
      end VoltageSource;

      model ActivePowerSensor
        extends Modelica.Icons.RotationalSensor;
        PositivePin s "Connect to the power source" annotation(Placement(visible = true, transformation(origin = {-100, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {-100, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
        PositivePin p "Connect to the load positive pin" annotation(Placement(visible = true, transformation(origin = {100, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {100, 0}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
        NegativePin n "Connect to the load negative pin" annotation(Placement(visible = true, transformation(origin = {0, 100}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {0, 100}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
        SI.ComplexVoltage v "Voltage across load";
        SI.ComplexCurrent i "Current through load";
        Modelica.Blocks.Interfaces.RealOutput P annotation(Placement(visible = true, transformation(origin = {4, -88}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {0, -100}, extent = {{-10, -10}, {10, 10}}, rotation = -90)));
        SI.ComplexPower S "Complex power flow through the component";
      equation
        s.v = p.v "Ideal sensor with zero impedance";
        v = p.v - n.v "Load voltage drop";
        s.i + p.i = Complex(0, 0) "Current going to load";
        n.i = Complex(0, 0) "No current into n pin";
        i = s.i "Current through load";
        S = v * Modelica.ComplexMath.conj(i) "Complex power";
        P = S.re;
        annotation(Icon(graphics = {Line(origin = {-79, 0}, points = {{-11, 0}, {9, 0}, {9, 0}}), Line(origin = {80, 1}, points = {{-10, -1}, {10, -1}}), Line(origin = {0, -80}, points = {{0, 10}, {0, -10}, {0, -10}}), Line(origin = {0, 80}, points = {{0, 10}, {0, -8}, {0, -10}})}, coordinateSystem(initialScale = 0.1)), uses(ScalableTestSuite(version = "1.7.1")));
      end ActivePowerSensor;

      partial model PartialImpedance "Generic complex impedance model"
        extends OnePort;
        SI.ComplexImpedance Z_ "Impedance - internal variable";
      equation
        v = Z_ * i;
        annotation(Icon(graphics = {Line(origin = {-75, 0}, points = {{-15, 0}, {15, 0}}), Line(origin = {75, 0}, points = {{15, 0}, {-15, 0}}), Rectangle(origin = {-1, 0}, extent = {{-59, 20}, {61, -20}})}, coordinateSystem(initialScale = 0.1)));
      end PartialImpedance;

      model Impedance "Generic complex impedance model"
        extends PartialImpedance(final Z_ = Z);
        parameter SI.ComplexImpedance Z "Fixed impedance";
        annotation(__OpenModelica_lateFlatten = true);
      end Impedance;

      model VariableResistor "Resistor model with variable resistance"
        extends PartialImpedance(Z_(re = R, im = 0));
        Modelica.Blocks.Interfaces.RealInput R annotation(Placement(visible = true, transformation(origin = {2, -60}, extent = {{-20, -20}, {20, 20}}, rotation = 0), iconTransformation(origin = {0, -30}, extent = {{-10, -10}, {10, 10}}, rotation = 90)));
        annotation(uses(ScalableTestSuite(version = "1.7.1")), Icon(graphics = {Line(origin = {-75, 0}, points = {{-15, 0}, {13, 0}}), Line(origin = {75, 0}, points = {{15, 0}, {-15, 0}}), Rectangle(origin = {-1, 0}, extent = {{-61, 20}, {61, -20}})}, coordinateSystem(initialScale = 0.1)));
      end VariableResistor;

      model VariableActivePowerLoad "Purely active load model with variable consumption"
        extends OnePort;
        Modelica.Blocks.Interfaces.RealInput P annotation(Placement(visible = true, transformation(origin = {2, -60}, extent = {{-20, -20}, {20, 20}}, rotation = 0), iconTransformation(origin = {0, -30}, extent = {{-10, -10}, {10, 10}}, rotation = 90)));
      equation
        v * Modelica.ComplexMath.conj(i) = Complex(P, 0);
        annotation(Icon(graphics = {Line(origin = {-75, 0}, points = {{-15, 0}, {13, 0}}), Line(origin = {75, 0}, points = {{15, 0}, {-15, 0}}), Rectangle(origin = {-1, 0}, extent = {{-61, 20}, {61, -20}})}, coordinateSystem(initialScale = 0.1)));
      end VariableActivePowerLoad;

      model LinearControlledLoad
        parameter SI.Voltage V_nom = 600 "Nominal voltage";
        parameter SI.Power P_nom "Nominal power";
        parameter SI.Time T = 0.1 "Time constant of power controller";
        final parameter SI.Resistance R_nom = V_nom ^ 2 / P_nom;
        PositivePin p annotation(Placement(visible = true, transformation(origin = {0, 100}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {0, 100}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
        VariableResistor R annotation(Placement(visible = true, transformation(origin = {-3.55271e-15, -8}, extent = {{-20, -20}, {20, 20}}, rotation = -90)));
        ScalableTestSuite.Electrical.DistributionSystemAC.Models.Internals.Ground ground1 annotation(Placement(visible = true, transformation(origin = {-3.55271e-15, -78}, extent = {{-20, -20}, {20, 20}}, rotation = 0)));
        ScalableTestSuite.Electrical.DistributionSystemAC.Models.Internals.ActivePowerSensor sensor annotation(Placement(visible = true, transformation(origin = {0, 60}, extent = {{-14, -14}, {14, 14}}, rotation = -90)));
        Modelica.Blocks.Continuous.Integrator controller(initType = Modelica.Blocks.Types.Init.InitialOutput, k = -R_nom / P_nom / T, y_start = R_nom) annotation(Placement(visible = true, transformation(origin = {-26, -8}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
        Modelica.Blocks.Math.Feedback feedback annotation(Placement(visible = true, transformation(origin = {-54, -8}, extent = {{-10, 10}, {10, -10}}, rotation = 0)));
        Modelica.Blocks.Sources.RealExpression powerSetpoint(y = P_nom) annotation(Placement(visible = true, transformation(origin = {-84, -8}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
      equation
        connect(sensor.s, p) annotation(Line(points = {{0, 74}, {0, 74}, {0, 100}, {0, 100}}, color = {0, 0, 255}));
        connect(sensor.p, R.p) annotation(Line(points = {{0, 46}, {0, 12}}, color = {0, 0, 255}));
        connect(sensor.n, R.n) annotation(Line(points = {{14, 60}, {40, 60}, {40, -28}, {0, -28}}, color = {0, 0, 255}));
        connect(feedback.u2, sensor.P) annotation(Line(points = {{-54, 0}, {-54, 60}, {-14, 60}}, color = {0, 0, 127}));
        connect(ground1.p, R.n) annotation(Line(points = {{0, -78}, {0, -78}, {0, -28}, {0, -28}}, color = {0, 0, 255}));
        connect(powerSetpoint.y, feedback.u1) annotation(Line(points = {{-72, -8}, {-64, -8}, {-64, -8}, {-62, -8}}, color = {0, 0, 127}));
        connect(controller.u, feedback.y) annotation(Line(points = {{-38, -8}, {-45, -8}}, color = {0, 0, 127}));
        connect(R.R, controller.y) annotation(Line(points = {{-6, -8}, {-15, -8}}, color = {0, 0, 127}));
        annotation(Icon(graphics = {Rectangle(origin = {-1, 0}, extent = {{-99, 100}, {101, -100}}), Line(origin = {0, 75}, points = {{0, 15}, {0, -15}}), Rectangle(origin = {-1, 4}, extent = {{-19, 56}, {21, -24}}), Line(origin = {0, -50}, points = {{0, 30}, {0, -10}, {0, -10}}), Line(origin = {-1, -60}, points = {{-19, 0}, {21, 0}}), Line(origin = {0, -68}, points = {{-10, 0}, {10, 0}}), Line(origin = {0, -76}, points = {{-4, 0}, {4, 0}})}, coordinateSystem(initialScale = 0.1)), __OpenModelica_lateFlatten = true);
      end LinearControlledLoad;

      model NoninearControlledLoad
        parameter SI.Voltage V_nom = 600 "Nominal voltage";
        parameter SI.Power P_nom "Nominal power";
        PositivePin p annotation(Placement(visible = true, transformation(origin = {0, 100}, extent = {{-10, -10}, {10, 10}}, rotation = 0), iconTransformation(origin = {0, 100}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
        VariableActivePowerLoad load annotation(Placement(visible = true, transformation(origin = {-3.55271e-15, -8}, extent = {{-20, -20}, {20, 20}}, rotation = -90)));
        Ground ground1 annotation(Placement(visible = true, transformation(origin = {-3.55271e-15, -60}, extent = {{-20, -20}, {20, 20}}, rotation = 0)));
        ActivePowerSensor sensor annotation(Placement(visible = true, transformation(origin = {0, 66}, extent = {{-14, -14}, {14, 14}}, rotation = -90)));
        Modelica.Blocks.Continuous.Integrator controller(initType = Modelica.Blocks.Types.Init.SteadyState, k = 1, y_start = P_nom / V_nom) annotation(Placement(visible = true, transformation(origin = {-26, -8}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
        Modelica.Blocks.Math.Feedback feedback annotation(Placement(visible = true, transformation(origin = {-54, -8}, extent = {{-10, 10}, {10, -10}}, rotation = 0)));
        Modelica.Blocks.Sources.RealExpression powerSetpoint(y = P_nom) annotation(Placement(visible = true, transformation(origin = {-84, -8}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
      equation
        connect(sensor.n, load.n) annotation(Line(points = {{14, 66}, {40, 66}, {40, -28}, {0, -28}, {0, -28}}, color = {0, 0, 255}));
        connect(powerSetpoint.y, feedback.u1) annotation(Line(points = {{-72, -8}, {-64, -8}, {-64, -8}, {-62, -8}}, color = {0, 0, 127}));
        connect(controller.u, feedback.y) annotation(Line(points = {{-38, -8}, {-45, -8}}, color = {0, 0, 127}));
        connect(feedback.u2, sensor.P) annotation(Line(points = {{-54, 0}, {-54, 66}, {-14, 66}}, color = {0, 0, 127}));
        connect(load.P, controller.y) annotation(Line(points = {{-6, -8}, {-15, -8}}, color = {0, 0, 127}));
        connect(load.n, ground1.p) annotation(Line(points = {{0, -28}, {0, -28}, {0, -60}, {0, -60}}, color = {0, 0, 255}));
        connect(sensor.n, load.p) annotation(Line(points = {{0, 52}, {0, 52}, {0, 12}, {0, 12}}, color = {0, 0, 255}));
        connect(p, sensor.p) annotation(Line(points = {{0, 100}, {0, 100}, {0, 80}, {0, 80}}, color = {0, 0, 255}));
        annotation(Icon(graphics = {Rectangle(origin = {-1, 0}, extent = {{-99, 100}, {101, -100}}), Line(origin = {0, 75}, points = {{0, 15}, {0, -15}}), Rectangle(origin = {-1, 4}, extent = {{-19, 56}, {21, -24}}), Line(origin = {0, -50}, points = {{0, 30}, {0, -10}, {0, -10}}), Line(origin = {-1, -60}, points = {{-19, 0}, {21, 0}}), Line(origin = {0, -68}, points = {{-10, 0}, {10, 0}}), Line(origin = {0, -76}, points = {{-4, 0}, {4, 0}})}, coordinateSystem(initialScale = 0.1)));
      end NoninearControlledLoad;
    end Internals;

    model DistributionSystemLinear
      parameter Integer N = 4 "Number of segments of the primary distribution line";
      parameter Integer M = N "Number of segments of each secondary distribution line";
      parameter Real alpha = 10 "Distribution line oversizing factor";
      parameter Real beta = 2 "Ratio between line inductance and line resistance";
      parameter SI.Resistance R_l = 100 "Resistance of a single load";
      parameter SI.Resistance R_d2 = R_l / (M ^ 2 * alpha) "Resistance of a secondary distribution segment";
      parameter SI.Resistance R_d1 = R_l / (M ^ 2 * N ^ 2 * alpha) "Resistance of a primary distribution segment";
      parameter Modelica.SIunits.Voltage V_ref = 600 "Reference source voltage";
      Internals.Impedance primary[N](each Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segments";
      Internals.Impedance secondary[N, M](each Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segments";
      Internals.LinearControlledLoad load[N, M](each V_nom = V_ref, each P_nom = V_ref ^ 2 / R_l) "Individual load resistors";
      Internals.Ground sourceGround "Source ground";
      Internals.VoltageSource V_source(V = V_ref) "Voltage source";
    equation
      connect(primary[1].p, V_source.p);
      connect(sourceGround.p, V_source.n);
      for i in 1:N - 1 loop
        connect(primary[i].n, primary[i + 1].p);
      end for;
      for i in 1:N loop
        connect(primary[i].n, secondary[i, 1].p);
        for j in 1:M - 1 loop
          connect(secondary[i, j].n, secondary[i, j + 1].p);
        end for;
        for j in 1:M loop
          connect(secondary[i, j].n, load[i, j].p);
        end for;
      end for;
      annotation(Documentation(info = "<html>
    <p>This model represnts an AC current distribution system, whose complexity depends on two parameters
    N and M.</p><p>A constant voltage source is connected to a primary resistive distribution line which is split into
    N segments, each with an impedance R_d1 + j*beta*R_d1. At the end of each segment, a secondary distribution
    line is attached with M elements each of impedance R_d2 + j*beta*R_d2. At the end of each secondary segment,
    a linear load connected, which is internally grounded on the other side.</p>
    <p>Each load measures its absorbed active power and adapts its internal resistance to match the required active
    power set point.
    </html>"));
    end DistributionSystemLinear;

    model DistributionSystemLinearIndividual
      impure function print
        input String s;
      algorithm
        Modelica.Utilities.Streams.print(s, "code.mo");
      end print;

      function printModel
      protected
        Integer N = 28 "Number of segments of the primary distribution line";
        Integer M = N "Number of segments of each secondary distribution line";
        Real alpha = 10 "Distribution line oversizing factor";
        Real beta = 2 "Ratio between line inductance and line resistance";
        Modelica.SIunits.Resistance R_l = 100 "Resistance of a single load";
      algorithm
        Modelica.Utilities.Files.removeFile("code.mo");
        print("model DistributionSystemLinearIndividual_N_" + String(N) + "_M_" + String(M));
        print("  parameter Integer N = " + String(N) + " \"Number of segments of the primary distribution line\";");
        print("  parameter Integer M = " + String(M) + " \"Number of segments of each secondary distribution line\";");
        print("  parameter Real alpha = " + String(alpha) + " \"Distribution line oversizing factor\";");
        print("  parameter Real beta = " + String(beta) + " \"Ratio between line inductance and line resistance\";");
        print("  parameter Modelica.SIunits.Resistance R_l = " + String(R_l) + " \"Resistance of a single load\";");
        print("  parameter Modelica.SIunits.Resistance R_d2 = R_l/(M^2*alpha) \"Resistance of a secondary distribution segment\";");
        print("  parameter Modelica.SIunits.Resistance R_d1 = R_l/(M^2*N^2*alpha) \"Resistance of a primary distribution segment\";");
        print("  parameter Modelica.SIunits.Voltage V_ref = 600 \"Reference source voltage\";");
        print("");
        for i in 1:N loop
          print("  Models.Internals.Impedance primary_" + String(i) + "(Z(re = R_d1, im = R_d1 * beta)) \"Primary distribution line segment\";");
          for j in 1:M loop
            print("  Models.Internals.Impedance secondary_" + String(i) + "_" + String(j) + "(Z(re = R_d2, im = R_d2 * beta)) \"Secondary distribution line segment\";");
            print("  Models.Internals.LinearControlledLoad load_" + String(i) + "_" + String(j) + "(V_nom = V_ref, P_nom = V_ref^2/R_l) \"Individual load resistor\";");
          end for;
        end for;
        print("  Models.Internals.Ground sourceGround \"Source ground\";");
        print("  Models.Internals.VoltageSource V_source(V = V_ref) \"Voltage source\";");
        print("equation");
        print("  connect(primary_1.p, V_source.p);");
        print("  connect(sourceGround.p, V_source.n);");
        for i in 1:N - 1 loop
          print("  connect(primary_" + String(i) + ".n, primary_" + String(i + 1) + ".p);");
        end for;
        for i in 1:N loop
          print("  connect(primary_" + String(i) + ".n, secondary_" + String(i) + "_1.p);");
          for j in 1:M - 1 loop
            print("  connect(secondary_" + String(i) + "_" + String(j) + ".n, secondary_" + String(i) + "_" + String(j + 1) + ".p);");
          end for;
          for j in 1:M loop
            print("  connect(secondary_" + String(i) + "_" + String(j) + ".n, load_" + String(i) + "_" + String(j) + ".p);");
          end for;
        end for;
        print("  annotation(experiment(StopTime = 1, Interval = 1e-3));");
        print("end DistributionSystemLinearIndividual_N_" + String(N) + "_M_" + String(M) + ";");
      end printModel;
    algorithm
      when terminal() then
        printModel();
      end when;
      annotation(experiment(StopTime = 1, Interval = 1e-3), Documentation(info = "<html>
    <p>This model generates Modelica code of models equivalent to DistributionSystemLinear which don&apos;t use arrays and for loops, but rather declare each model and each connection individually.</p>
    <p>This model can be used to check the overhead of instantiating large numbesr of individual models compared to arrays, and also to check the ability of compilers to factor out the code of instances of the same component.</p>
    </html>"));
    end DistributionSystemLinearIndividual;
  end Models;

  package Verification
    model DistributionSystemLinear_N_2_M_2
      extends Models.DistributionSystemLinear(N = 2, M = 2);
      annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinear_N_2_M_2;

    model DistributionSystemLinearIndividual_N_2_M_2 "Automatically generated by DistributionSystemLinearIndividual"
      parameter Integer N = 2 "Number of segments of the primary distribution line";
      parameter Integer M = 2 "Number of segments of each secondary distribution line";
      parameter Real alpha = 10 "Distribution line oversizing factor";
      parameter Real beta = 2 "Ratio between line inductance and line resistance";
      parameter Modelica.SIunits.Resistance R_l = 100 "Resistance of a single load";
      parameter Modelica.SIunits.Resistance R_d2 = R_l/(M^2*alpha) "Resistance of a secondary distribution segment";
      parameter Modelica.SIunits.Resistance R_d1 = R_l/(M^2*N^2*alpha) "Resistance of a primary distribution segment";
      parameter Modelica.SIunits.Voltage V_ref = 600 "Reference source voltage";
    
      Models.Internals.Impedance primary_1(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_1_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_1(V_nom = V_ref, P_nom = V_ref^2/R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_2(V_nom = V_ref, P_nom = V_ref^2/R_l) "Individual load resistor";
      Models.Internals.Impedance primary_2(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_2_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_1(V_nom = V_ref, P_nom = V_ref^2/R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_2(V_nom = V_ref, P_nom = V_ref^2/R_l) "Individual load resistor";
      Models.Internals.Ground sourceGround "Source ground";
      Models.Internals.VoltageSource V_source(V = V_ref) "Voltage source";
    equation
      connect(primary_1.p, V_source.p);
      connect(sourceGround.p, V_source.n);
      connect(primary_1.n, primary_2.p);
      connect(primary_1.n, secondary_1_1.p);
      connect(secondary_1_1.n, secondary_1_2.p);
      connect(secondary_1_1.n, load_1_1.p);
      connect(secondary_1_2.n, load_1_2.p);
      connect(primary_2.n, secondary_2_1.p);
      connect(secondary_2_1.n, secondary_2_2.p);
      connect(secondary_2_1.n, load_2_1.p);
      connect(secondary_2_2.n, load_2_2.p);
      annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinearIndividual_N_2_M_2;
  
  
    model LinearControlledLoad
      Models.Internals.LinearControlledLoad linearControlledLoad(P_nom = 3600)  annotation(Placement(visible = true, transformation(origin = {0, 8.88178e-16}, extent = {{-20, -20}, {20, 20}}, rotation = 0)));
  Models.Internals.VoltageSource voltageSource(V = 650)  annotation(Placement(visible = true, transformation(origin = {-50, 3.55271e-15}, extent = {{-20, -20}, {20, 20}}, rotation = -90)));
  Models.Internals.Ground ground annotation(Placement(visible = true, transformation(origin = {-50, -40}, extent = {{-20, -20}, {20, 20}}, rotation = 0)));
      equation
      connect(voltageSource.p, linearControlledLoad.p) annotation(Line(points = {{-50, 20}, {-50, 40}, {0, 40}, {0, 20}}, color = {0, 0, 255}));
      connect(ground.p, voltageSource.n) annotation(Line(points = {{-50, -40}, {-50, -40}, {-50, -20}, {-50, -20}}, color = {0, 0, 255}));
      annotation(Documentation(info="<html>Test of the Linear Controlled Load Model. The applied voltage is slightly higher than the nominal one, so the controller reacts by adapting the resistance to restore the nominal power consumption</html>"));
    end LinearControlledLoad;

    model VariableResistor "Checks the behaviour of the variable resistor and power flow sensor"
  Models.Internals.Ground ground annotation(Placement(visible = true, transformation(origin = {1.77636e-15, -56}, extent = {{-20, -20}, {20, 20}}, rotation = 0)));
  Models.Internals.VariableResistor resistor annotation(Placement(visible = true, transformation(origin = {3.55271e-15, -16}, extent = {{-20, -20}, {20, 20}}, rotation = -90)));
  Models.Internals.VoltageSource voltageSource(V = 600) annotation(Placement(visible = true, transformation(origin = {40, 4}, extent = {{-20, -20}, {20, 20}}, rotation = -90)));
  Modelica.Blocks.Sources.RealExpression resistance(y = 300.0) annotation(Placement(visible = true, transformation(origin = {-52, -16}, extent = {{-10, -10}, {10, 10}}, rotation = 0)));
  Models.Internals.ActivePowerSensor sensor annotation(Placement(visible = true, transformation(origin = {7.10543e-15, 44}, extent = {{-20, -20}, {20, 20}}, rotation = -90)));
      equation
      connect(sensor.n, resistor.n) annotation(Line(points = {{20, 44}, {22, 44}, {22, 44}, {24, 44}, {24, 44}, {28, 44}, {28, -36}, {1.49012e-07, -36}, {1.49012e-07, -36}, {1.49012e-07, -36}, {1.49012e-07, -36}}, color = {0, 0, 255}));
      connect(voltageSource.p, sensor.s) annotation(Line(points = {{40, 24}, {40, 24}, {40, 24}, {40, 24}, {40, 24}, {40, 24}, {40, 64}, {0, 64}, {0, 64}, {0, 64}, {0, 64}}, color = {0, 0, 255}));
      connect(resistor.p, sensor.p) annotation(Line(points = {{3.55271e-15, 4}, {3.55271e-15, 24}}, color = {0, 0, 255}));
      connect(resistance.y, resistor.R) annotation(Line(points = {{-41, -16}, {-6, -16}}, color = {0, 0, 127}));
      connect(voltageSource.n, ground.p) annotation(Line(points = {{40, -16}, {40, -36}, {40, -36}, {40, -56}, {20, -56}, {20, -56}, {10, -56}, {10, -56}, {0, -56}}, color = {0, 0, 255}));
      connect(resistor.n, ground.p) annotation(Line(points = {{3.55271e-15, -36}, {3.55271e-15, -56}}, color = {0, 0, 255}));
      annotation(Documentation(info="<html>Test of the Variable Resistor model. A voltage of 600 V is applied to a resistor of 300 Ohms. The current is 2 A and the power is 1200 W.</html>"));
    end VariableResistor;
  end Verification;

  package ScaledExperiments
    extends Modelica.Icons.ExamplesPackage;

    model DistributionSystemLinear_N_10_M_10
      extends Models.DistributionSystemLinear(N = 10, M = 10);
      annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinear_N_10_M_10;

    model DistributionSystemLinear_N_14_M_14
      extends Models.DistributionSystemLinear(N = 14, M = 14);
      annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinear_N_14_M_14;

    model DistributionSystemLinear_N_20_M_20
      extends Models.DistributionSystemLinear(N = 20, M = 20);
      annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinear_N_20_M_20;

    model DistributionSystemLinear_N_28_M_28
      extends Models.DistributionSystemLinear(N = 28, M = 28);
      annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinear_N_28_M_28;

    model DistributionSystemLinear_N_40_M_40
      extends Models.DistributionSystemLinear(N = 40, M = 40);
      // annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinear_N_40_M_40;

    model DistributionSystemLinear_N_56_M_56
      extends Models.DistributionSystemLinear(N = 56, M = 56);
      // annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinear_N_56_M_56;

    model DistributionSystemLinear_N_80_M_80
      extends Models.DistributionSystemLinear(N = 80, M = 80);
      //annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinear_N_80_M_80;

    model DistributionSystemLinear_N_112_M_112
      extends Models.DistributionSystemLinear(N = 112, M = 112);
      //annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinear_N_112_M_112;

    model DistributionSystemLinear_N_160_M_160
      extends Models.DistributionSystemLinear(N = 160, M = 160);
      //annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinear_N_160_M_160;

    model DistributionSystemLinearIndividual_N_10_M_10
      parameter Integer N = 10 "Number of segments of the primary distribution line";
      parameter Integer M = 10 "Number of segments of each secondary distribution line";
      parameter Real alpha = 10 "Distribution line oversizing factor";
      parameter Real beta = 2 "Ratio between line inductance and line resistance";
      parameter Modelica.SIunits.Resistance R_l = 100 "Resistance of a single load";
      parameter Modelica.SIunits.Resistance R_d2 = R_l / (M ^ 2 * alpha) "Resistance of a secondary distribution segment";
      parameter Modelica.SIunits.Resistance R_d1 = R_l / (M ^ 2 * N ^ 2 * alpha) "Resistance of a primary distribution segment";
      parameter Modelica.SIunits.Voltage V_ref = 600 "Reference source voltage";
      Models.Internals.Impedance primary_1(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_1_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_2(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_2_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_3(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_3_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_4(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_4_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_5(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_5_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_6(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_6_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_7(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_7_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_8(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_8_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_9(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_9_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_10(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_10_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Ground sourceGround "Source ground";
      Models.Internals.VoltageSource V_source(V = V_ref) "Voltage source";
    equation
      connect(primary_1.p, V_source.p);
      connect(sourceGround.p, V_source.n);
      connect(primary_1.n, primary_2.p);
      connect(primary_2.n, primary_3.p);
      connect(primary_3.n, primary_4.p);
      connect(primary_4.n, primary_5.p);
      connect(primary_5.n, primary_6.p);
      connect(primary_6.n, primary_7.p);
      connect(primary_7.n, primary_8.p);
      connect(primary_8.n, primary_9.p);
      connect(primary_9.n, primary_10.p);
      connect(primary_1.n, secondary_1_1.p);
      connect(secondary_1_1.n, secondary_1_2.p);
      connect(secondary_1_2.n, secondary_1_3.p);
      connect(secondary_1_3.n, secondary_1_4.p);
      connect(secondary_1_4.n, secondary_1_5.p);
      connect(secondary_1_5.n, secondary_1_6.p);
      connect(secondary_1_6.n, secondary_1_7.p);
      connect(secondary_1_7.n, secondary_1_8.p);
      connect(secondary_1_8.n, secondary_1_9.p);
      connect(secondary_1_9.n, secondary_1_10.p);
      connect(secondary_1_1.n, load_1_1.p);
      connect(secondary_1_2.n, load_1_2.p);
      connect(secondary_1_3.n, load_1_3.p);
      connect(secondary_1_4.n, load_1_4.p);
      connect(secondary_1_5.n, load_1_5.p);
      connect(secondary_1_6.n, load_1_6.p);
      connect(secondary_1_7.n, load_1_7.p);
      connect(secondary_1_8.n, load_1_8.p);
      connect(secondary_1_9.n, load_1_9.p);
      connect(secondary_1_10.n, load_1_10.p);
      connect(primary_2.n, secondary_2_1.p);
      connect(secondary_2_1.n, secondary_2_2.p);
      connect(secondary_2_2.n, secondary_2_3.p);
      connect(secondary_2_3.n, secondary_2_4.p);
      connect(secondary_2_4.n, secondary_2_5.p);
      connect(secondary_2_5.n, secondary_2_6.p);
      connect(secondary_2_6.n, secondary_2_7.p);
      connect(secondary_2_7.n, secondary_2_8.p);
      connect(secondary_2_8.n, secondary_2_9.p);
      connect(secondary_2_9.n, secondary_2_10.p);
      connect(secondary_2_1.n, load_2_1.p);
      connect(secondary_2_2.n, load_2_2.p);
      connect(secondary_2_3.n, load_2_3.p);
      connect(secondary_2_4.n, load_2_4.p);
      connect(secondary_2_5.n, load_2_5.p);
      connect(secondary_2_6.n, load_2_6.p);
      connect(secondary_2_7.n, load_2_7.p);
      connect(secondary_2_8.n, load_2_8.p);
      connect(secondary_2_9.n, load_2_9.p);
      connect(secondary_2_10.n, load_2_10.p);
      connect(primary_3.n, secondary_3_1.p);
      connect(secondary_3_1.n, secondary_3_2.p);
      connect(secondary_3_2.n, secondary_3_3.p);
      connect(secondary_3_3.n, secondary_3_4.p);
      connect(secondary_3_4.n, secondary_3_5.p);
      connect(secondary_3_5.n, secondary_3_6.p);
      connect(secondary_3_6.n, secondary_3_7.p);
      connect(secondary_3_7.n, secondary_3_8.p);
      connect(secondary_3_8.n, secondary_3_9.p);
      connect(secondary_3_9.n, secondary_3_10.p);
      connect(secondary_3_1.n, load_3_1.p);
      connect(secondary_3_2.n, load_3_2.p);
      connect(secondary_3_3.n, load_3_3.p);
      connect(secondary_3_4.n, load_3_4.p);
      connect(secondary_3_5.n, load_3_5.p);
      connect(secondary_3_6.n, load_3_6.p);
      connect(secondary_3_7.n, load_3_7.p);
      connect(secondary_3_8.n, load_3_8.p);
      connect(secondary_3_9.n, load_3_9.p);
      connect(secondary_3_10.n, load_3_10.p);
      connect(primary_4.n, secondary_4_1.p);
      connect(secondary_4_1.n, secondary_4_2.p);
      connect(secondary_4_2.n, secondary_4_3.p);
      connect(secondary_4_3.n, secondary_4_4.p);
      connect(secondary_4_4.n, secondary_4_5.p);
      connect(secondary_4_5.n, secondary_4_6.p);
      connect(secondary_4_6.n, secondary_4_7.p);
      connect(secondary_4_7.n, secondary_4_8.p);
      connect(secondary_4_8.n, secondary_4_9.p);
      connect(secondary_4_9.n, secondary_4_10.p);
      connect(secondary_4_1.n, load_4_1.p);
      connect(secondary_4_2.n, load_4_2.p);
      connect(secondary_4_3.n, load_4_3.p);
      connect(secondary_4_4.n, load_4_4.p);
      connect(secondary_4_5.n, load_4_5.p);
      connect(secondary_4_6.n, load_4_6.p);
      connect(secondary_4_7.n, load_4_7.p);
      connect(secondary_4_8.n, load_4_8.p);
      connect(secondary_4_9.n, load_4_9.p);
      connect(secondary_4_10.n, load_4_10.p);
      connect(primary_5.n, secondary_5_1.p);
      connect(secondary_5_1.n, secondary_5_2.p);
      connect(secondary_5_2.n, secondary_5_3.p);
      connect(secondary_5_3.n, secondary_5_4.p);
      connect(secondary_5_4.n, secondary_5_5.p);
      connect(secondary_5_5.n, secondary_5_6.p);
      connect(secondary_5_6.n, secondary_5_7.p);
      connect(secondary_5_7.n, secondary_5_8.p);
      connect(secondary_5_8.n, secondary_5_9.p);
      connect(secondary_5_9.n, secondary_5_10.p);
      connect(secondary_5_1.n, load_5_1.p);
      connect(secondary_5_2.n, load_5_2.p);
      connect(secondary_5_3.n, load_5_3.p);
      connect(secondary_5_4.n, load_5_4.p);
      connect(secondary_5_5.n, load_5_5.p);
      connect(secondary_5_6.n, load_5_6.p);
      connect(secondary_5_7.n, load_5_7.p);
      connect(secondary_5_8.n, load_5_8.p);
      connect(secondary_5_9.n, load_5_9.p);
      connect(secondary_5_10.n, load_5_10.p);
      connect(primary_6.n, secondary_6_1.p);
      connect(secondary_6_1.n, secondary_6_2.p);
      connect(secondary_6_2.n, secondary_6_3.p);
      connect(secondary_6_3.n, secondary_6_4.p);
      connect(secondary_6_4.n, secondary_6_5.p);
      connect(secondary_6_5.n, secondary_6_6.p);
      connect(secondary_6_6.n, secondary_6_7.p);
      connect(secondary_6_7.n, secondary_6_8.p);
      connect(secondary_6_8.n, secondary_6_9.p);
      connect(secondary_6_9.n, secondary_6_10.p);
      connect(secondary_6_1.n, load_6_1.p);
      connect(secondary_6_2.n, load_6_2.p);
      connect(secondary_6_3.n, load_6_3.p);
      connect(secondary_6_4.n, load_6_4.p);
      connect(secondary_6_5.n, load_6_5.p);
      connect(secondary_6_6.n, load_6_6.p);
      connect(secondary_6_7.n, load_6_7.p);
      connect(secondary_6_8.n, load_6_8.p);
      connect(secondary_6_9.n, load_6_9.p);
      connect(secondary_6_10.n, load_6_10.p);
      connect(primary_7.n, secondary_7_1.p);
      connect(secondary_7_1.n, secondary_7_2.p);
      connect(secondary_7_2.n, secondary_7_3.p);
      connect(secondary_7_3.n, secondary_7_4.p);
      connect(secondary_7_4.n, secondary_7_5.p);
      connect(secondary_7_5.n, secondary_7_6.p);
      connect(secondary_7_6.n, secondary_7_7.p);
      connect(secondary_7_7.n, secondary_7_8.p);
      connect(secondary_7_8.n, secondary_7_9.p);
      connect(secondary_7_9.n, secondary_7_10.p);
      connect(secondary_7_1.n, load_7_1.p);
      connect(secondary_7_2.n, load_7_2.p);
      connect(secondary_7_3.n, load_7_3.p);
      connect(secondary_7_4.n, load_7_4.p);
      connect(secondary_7_5.n, load_7_5.p);
      connect(secondary_7_6.n, load_7_6.p);
      connect(secondary_7_7.n, load_7_7.p);
      connect(secondary_7_8.n, load_7_8.p);
      connect(secondary_7_9.n, load_7_9.p);
      connect(secondary_7_10.n, load_7_10.p);
      connect(primary_8.n, secondary_8_1.p);
      connect(secondary_8_1.n, secondary_8_2.p);
      connect(secondary_8_2.n, secondary_8_3.p);
      connect(secondary_8_3.n, secondary_8_4.p);
      connect(secondary_8_4.n, secondary_8_5.p);
      connect(secondary_8_5.n, secondary_8_6.p);
      connect(secondary_8_6.n, secondary_8_7.p);
      connect(secondary_8_7.n, secondary_8_8.p);
      connect(secondary_8_8.n, secondary_8_9.p);
      connect(secondary_8_9.n, secondary_8_10.p);
      connect(secondary_8_1.n, load_8_1.p);
      connect(secondary_8_2.n, load_8_2.p);
      connect(secondary_8_3.n, load_8_3.p);
      connect(secondary_8_4.n, load_8_4.p);
      connect(secondary_8_5.n, load_8_5.p);
      connect(secondary_8_6.n, load_8_6.p);
      connect(secondary_8_7.n, load_8_7.p);
      connect(secondary_8_8.n, load_8_8.p);
      connect(secondary_8_9.n, load_8_9.p);
      connect(secondary_8_10.n, load_8_10.p);
      connect(primary_9.n, secondary_9_1.p);
      connect(secondary_9_1.n, secondary_9_2.p);
      connect(secondary_9_2.n, secondary_9_3.p);
      connect(secondary_9_3.n, secondary_9_4.p);
      connect(secondary_9_4.n, secondary_9_5.p);
      connect(secondary_9_5.n, secondary_9_6.p);
      connect(secondary_9_6.n, secondary_9_7.p);
      connect(secondary_9_7.n, secondary_9_8.p);
      connect(secondary_9_8.n, secondary_9_9.p);
      connect(secondary_9_9.n, secondary_9_10.p);
      connect(secondary_9_1.n, load_9_1.p);
      connect(secondary_9_2.n, load_9_2.p);
      connect(secondary_9_3.n, load_9_3.p);
      connect(secondary_9_4.n, load_9_4.p);
      connect(secondary_9_5.n, load_9_5.p);
      connect(secondary_9_6.n, load_9_6.p);
      connect(secondary_9_7.n, load_9_7.p);
      connect(secondary_9_8.n, load_9_8.p);
      connect(secondary_9_9.n, load_9_9.p);
      connect(secondary_9_10.n, load_9_10.p);
      connect(primary_10.n, secondary_10_1.p);
      connect(secondary_10_1.n, secondary_10_2.p);
      connect(secondary_10_2.n, secondary_10_3.p);
      connect(secondary_10_3.n, secondary_10_4.p);
      connect(secondary_10_4.n, secondary_10_5.p);
      connect(secondary_10_5.n, secondary_10_6.p);
      connect(secondary_10_6.n, secondary_10_7.p);
      connect(secondary_10_7.n, secondary_10_8.p);
      connect(secondary_10_8.n, secondary_10_9.p);
      connect(secondary_10_9.n, secondary_10_10.p);
      connect(secondary_10_1.n, load_10_1.p);
      connect(secondary_10_2.n, load_10_2.p);
      connect(secondary_10_3.n, load_10_3.p);
      connect(secondary_10_4.n, load_10_4.p);
      connect(secondary_10_5.n, load_10_5.p);
      connect(secondary_10_6.n, load_10_6.p);
      connect(secondary_10_7.n, load_10_7.p);
      connect(secondary_10_8.n, load_10_8.p);
      connect(secondary_10_9.n, load_10_9.p);
      connect(secondary_10_10.n, load_10_10.p);
      annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinearIndividual_N_10_M_10;

    model DistributionSystemLinearIndividual_N_14_M_14
      parameter Integer N = 14 "Number of segments of the primary distribution line";
      parameter Integer M = 14 "Number of segments of each secondary distribution line";
      parameter Real alpha = 10 "Distribution line oversizing factor";
      parameter Real beta = 2 "Ratio between line inductance and line resistance";
      parameter Modelica.SIunits.Resistance R_l = 100 "Resistance of a single load";
      parameter Modelica.SIunits.Resistance R_d2 = R_l / (M ^ 2 * alpha) "Resistance of a secondary distribution segment";
      parameter Modelica.SIunits.Resistance R_d1 = R_l / (M ^ 2 * N ^ 2 * alpha) "Resistance of a primary distribution segment";
      parameter Modelica.SIunits.Voltage V_ref = 600 "Reference source voltage";
      Models.Internals.Impedance primary_1(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_1_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_2(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_2_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_3(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_3_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_4(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_4_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_5(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_5_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_6(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_6_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_7(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_7_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_8(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_8_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_9(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_9_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_10(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_10_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_11(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_11_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_12(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_12_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_13(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_13_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_14(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_14_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Ground sourceGround "Source ground";
      Models.Internals.VoltageSource V_source(V = V_ref) "Voltage source";
    equation
      connect(primary_1.p, V_source.p);
      connect(sourceGround.p, V_source.n);
      connect(primary_1.n, primary_2.p);
      connect(primary_2.n, primary_3.p);
      connect(primary_3.n, primary_4.p);
      connect(primary_4.n, primary_5.p);
      connect(primary_5.n, primary_6.p);
      connect(primary_6.n, primary_7.p);
      connect(primary_7.n, primary_8.p);
      connect(primary_8.n, primary_9.p);
      connect(primary_9.n, primary_10.p);
      connect(primary_10.n, primary_11.p);
      connect(primary_11.n, primary_12.p);
      connect(primary_12.n, primary_13.p);
      connect(primary_13.n, primary_14.p);
      connect(primary_1.n, secondary_1_1.p);
      connect(secondary_1_1.n, secondary_1_2.p);
      connect(secondary_1_2.n, secondary_1_3.p);
      connect(secondary_1_3.n, secondary_1_4.p);
      connect(secondary_1_4.n, secondary_1_5.p);
      connect(secondary_1_5.n, secondary_1_6.p);
      connect(secondary_1_6.n, secondary_1_7.p);
      connect(secondary_1_7.n, secondary_1_8.p);
      connect(secondary_1_8.n, secondary_1_9.p);
      connect(secondary_1_9.n, secondary_1_10.p);
      connect(secondary_1_10.n, secondary_1_11.p);
      connect(secondary_1_11.n, secondary_1_12.p);
      connect(secondary_1_12.n, secondary_1_13.p);
      connect(secondary_1_13.n, secondary_1_14.p);
      connect(secondary_1_1.n, load_1_1.p);
      connect(secondary_1_2.n, load_1_2.p);
      connect(secondary_1_3.n, load_1_3.p);
      connect(secondary_1_4.n, load_1_4.p);
      connect(secondary_1_5.n, load_1_5.p);
      connect(secondary_1_6.n, load_1_6.p);
      connect(secondary_1_7.n, load_1_7.p);
      connect(secondary_1_8.n, load_1_8.p);
      connect(secondary_1_9.n, load_1_9.p);
      connect(secondary_1_10.n, load_1_10.p);
      connect(secondary_1_11.n, load_1_11.p);
      connect(secondary_1_12.n, load_1_12.p);
      connect(secondary_1_13.n, load_1_13.p);
      connect(secondary_1_14.n, load_1_14.p);
      connect(primary_2.n, secondary_2_1.p);
      connect(secondary_2_1.n, secondary_2_2.p);
      connect(secondary_2_2.n, secondary_2_3.p);
      connect(secondary_2_3.n, secondary_2_4.p);
      connect(secondary_2_4.n, secondary_2_5.p);
      connect(secondary_2_5.n, secondary_2_6.p);
      connect(secondary_2_6.n, secondary_2_7.p);
      connect(secondary_2_7.n, secondary_2_8.p);
      connect(secondary_2_8.n, secondary_2_9.p);
      connect(secondary_2_9.n, secondary_2_10.p);
      connect(secondary_2_10.n, secondary_2_11.p);
      connect(secondary_2_11.n, secondary_2_12.p);
      connect(secondary_2_12.n, secondary_2_13.p);
      connect(secondary_2_13.n, secondary_2_14.p);
      connect(secondary_2_1.n, load_2_1.p);
      connect(secondary_2_2.n, load_2_2.p);
      connect(secondary_2_3.n, load_2_3.p);
      connect(secondary_2_4.n, load_2_4.p);
      connect(secondary_2_5.n, load_2_5.p);
      connect(secondary_2_6.n, load_2_6.p);
      connect(secondary_2_7.n, load_2_7.p);
      connect(secondary_2_8.n, load_2_8.p);
      connect(secondary_2_9.n, load_2_9.p);
      connect(secondary_2_10.n, load_2_10.p);
      connect(secondary_2_11.n, load_2_11.p);
      connect(secondary_2_12.n, load_2_12.p);
      connect(secondary_2_13.n, load_2_13.p);
      connect(secondary_2_14.n, load_2_14.p);
      connect(primary_3.n, secondary_3_1.p);
      connect(secondary_3_1.n, secondary_3_2.p);
      connect(secondary_3_2.n, secondary_3_3.p);
      connect(secondary_3_3.n, secondary_3_4.p);
      connect(secondary_3_4.n, secondary_3_5.p);
      connect(secondary_3_5.n, secondary_3_6.p);
      connect(secondary_3_6.n, secondary_3_7.p);
      connect(secondary_3_7.n, secondary_3_8.p);
      connect(secondary_3_8.n, secondary_3_9.p);
      connect(secondary_3_9.n, secondary_3_10.p);
      connect(secondary_3_10.n, secondary_3_11.p);
      connect(secondary_3_11.n, secondary_3_12.p);
      connect(secondary_3_12.n, secondary_3_13.p);
      connect(secondary_3_13.n, secondary_3_14.p);
      connect(secondary_3_1.n, load_3_1.p);
      connect(secondary_3_2.n, load_3_2.p);
      connect(secondary_3_3.n, load_3_3.p);
      connect(secondary_3_4.n, load_3_4.p);
      connect(secondary_3_5.n, load_3_5.p);
      connect(secondary_3_6.n, load_3_6.p);
      connect(secondary_3_7.n, load_3_7.p);
      connect(secondary_3_8.n, load_3_8.p);
      connect(secondary_3_9.n, load_3_9.p);
      connect(secondary_3_10.n, load_3_10.p);
      connect(secondary_3_11.n, load_3_11.p);
      connect(secondary_3_12.n, load_3_12.p);
      connect(secondary_3_13.n, load_3_13.p);
      connect(secondary_3_14.n, load_3_14.p);
      connect(primary_4.n, secondary_4_1.p);
      connect(secondary_4_1.n, secondary_4_2.p);
      connect(secondary_4_2.n, secondary_4_3.p);
      connect(secondary_4_3.n, secondary_4_4.p);
      connect(secondary_4_4.n, secondary_4_5.p);
      connect(secondary_4_5.n, secondary_4_6.p);
      connect(secondary_4_6.n, secondary_4_7.p);
      connect(secondary_4_7.n, secondary_4_8.p);
      connect(secondary_4_8.n, secondary_4_9.p);
      connect(secondary_4_9.n, secondary_4_10.p);
      connect(secondary_4_10.n, secondary_4_11.p);
      connect(secondary_4_11.n, secondary_4_12.p);
      connect(secondary_4_12.n, secondary_4_13.p);
      connect(secondary_4_13.n, secondary_4_14.p);
      connect(secondary_4_1.n, load_4_1.p);
      connect(secondary_4_2.n, load_4_2.p);
      connect(secondary_4_3.n, load_4_3.p);
      connect(secondary_4_4.n, load_4_4.p);
      connect(secondary_4_5.n, load_4_5.p);
      connect(secondary_4_6.n, load_4_6.p);
      connect(secondary_4_7.n, load_4_7.p);
      connect(secondary_4_8.n, load_4_8.p);
      connect(secondary_4_9.n, load_4_9.p);
      connect(secondary_4_10.n, load_4_10.p);
      connect(secondary_4_11.n, load_4_11.p);
      connect(secondary_4_12.n, load_4_12.p);
      connect(secondary_4_13.n, load_4_13.p);
      connect(secondary_4_14.n, load_4_14.p);
      connect(primary_5.n, secondary_5_1.p);
      connect(secondary_5_1.n, secondary_5_2.p);
      connect(secondary_5_2.n, secondary_5_3.p);
      connect(secondary_5_3.n, secondary_5_4.p);
      connect(secondary_5_4.n, secondary_5_5.p);
      connect(secondary_5_5.n, secondary_5_6.p);
      connect(secondary_5_6.n, secondary_5_7.p);
      connect(secondary_5_7.n, secondary_5_8.p);
      connect(secondary_5_8.n, secondary_5_9.p);
      connect(secondary_5_9.n, secondary_5_10.p);
      connect(secondary_5_10.n, secondary_5_11.p);
      connect(secondary_5_11.n, secondary_5_12.p);
      connect(secondary_5_12.n, secondary_5_13.p);
      connect(secondary_5_13.n, secondary_5_14.p);
      connect(secondary_5_1.n, load_5_1.p);
      connect(secondary_5_2.n, load_5_2.p);
      connect(secondary_5_3.n, load_5_3.p);
      connect(secondary_5_4.n, load_5_4.p);
      connect(secondary_5_5.n, load_5_5.p);
      connect(secondary_5_6.n, load_5_6.p);
      connect(secondary_5_7.n, load_5_7.p);
      connect(secondary_5_8.n, load_5_8.p);
      connect(secondary_5_9.n, load_5_9.p);
      connect(secondary_5_10.n, load_5_10.p);
      connect(secondary_5_11.n, load_5_11.p);
      connect(secondary_5_12.n, load_5_12.p);
      connect(secondary_5_13.n, load_5_13.p);
      connect(secondary_5_14.n, load_5_14.p);
      connect(primary_6.n, secondary_6_1.p);
      connect(secondary_6_1.n, secondary_6_2.p);
      connect(secondary_6_2.n, secondary_6_3.p);
      connect(secondary_6_3.n, secondary_6_4.p);
      connect(secondary_6_4.n, secondary_6_5.p);
      connect(secondary_6_5.n, secondary_6_6.p);
      connect(secondary_6_6.n, secondary_6_7.p);
      connect(secondary_6_7.n, secondary_6_8.p);
      connect(secondary_6_8.n, secondary_6_9.p);
      connect(secondary_6_9.n, secondary_6_10.p);
      connect(secondary_6_10.n, secondary_6_11.p);
      connect(secondary_6_11.n, secondary_6_12.p);
      connect(secondary_6_12.n, secondary_6_13.p);
      connect(secondary_6_13.n, secondary_6_14.p);
      connect(secondary_6_1.n, load_6_1.p);
      connect(secondary_6_2.n, load_6_2.p);
      connect(secondary_6_3.n, load_6_3.p);
      connect(secondary_6_4.n, load_6_4.p);
      connect(secondary_6_5.n, load_6_5.p);
      connect(secondary_6_6.n, load_6_6.p);
      connect(secondary_6_7.n, load_6_7.p);
      connect(secondary_6_8.n, load_6_8.p);
      connect(secondary_6_9.n, load_6_9.p);
      connect(secondary_6_10.n, load_6_10.p);
      connect(secondary_6_11.n, load_6_11.p);
      connect(secondary_6_12.n, load_6_12.p);
      connect(secondary_6_13.n, load_6_13.p);
      connect(secondary_6_14.n, load_6_14.p);
      connect(primary_7.n, secondary_7_1.p);
      connect(secondary_7_1.n, secondary_7_2.p);
      connect(secondary_7_2.n, secondary_7_3.p);
      connect(secondary_7_3.n, secondary_7_4.p);
      connect(secondary_7_4.n, secondary_7_5.p);
      connect(secondary_7_5.n, secondary_7_6.p);
      connect(secondary_7_6.n, secondary_7_7.p);
      connect(secondary_7_7.n, secondary_7_8.p);
      connect(secondary_7_8.n, secondary_7_9.p);
      connect(secondary_7_9.n, secondary_7_10.p);
      connect(secondary_7_10.n, secondary_7_11.p);
      connect(secondary_7_11.n, secondary_7_12.p);
      connect(secondary_7_12.n, secondary_7_13.p);
      connect(secondary_7_13.n, secondary_7_14.p);
      connect(secondary_7_1.n, load_7_1.p);
      connect(secondary_7_2.n, load_7_2.p);
      connect(secondary_7_3.n, load_7_3.p);
      connect(secondary_7_4.n, load_7_4.p);
      connect(secondary_7_5.n, load_7_5.p);
      connect(secondary_7_6.n, load_7_6.p);
      connect(secondary_7_7.n, load_7_7.p);
      connect(secondary_7_8.n, load_7_8.p);
      connect(secondary_7_9.n, load_7_9.p);
      connect(secondary_7_10.n, load_7_10.p);
      connect(secondary_7_11.n, load_7_11.p);
      connect(secondary_7_12.n, load_7_12.p);
      connect(secondary_7_13.n, load_7_13.p);
      connect(secondary_7_14.n, load_7_14.p);
      connect(primary_8.n, secondary_8_1.p);
      connect(secondary_8_1.n, secondary_8_2.p);
      connect(secondary_8_2.n, secondary_8_3.p);
      connect(secondary_8_3.n, secondary_8_4.p);
      connect(secondary_8_4.n, secondary_8_5.p);
      connect(secondary_8_5.n, secondary_8_6.p);
      connect(secondary_8_6.n, secondary_8_7.p);
      connect(secondary_8_7.n, secondary_8_8.p);
      connect(secondary_8_8.n, secondary_8_9.p);
      connect(secondary_8_9.n, secondary_8_10.p);
      connect(secondary_8_10.n, secondary_8_11.p);
      connect(secondary_8_11.n, secondary_8_12.p);
      connect(secondary_8_12.n, secondary_8_13.p);
      connect(secondary_8_13.n, secondary_8_14.p);
      connect(secondary_8_1.n, load_8_1.p);
      connect(secondary_8_2.n, load_8_2.p);
      connect(secondary_8_3.n, load_8_3.p);
      connect(secondary_8_4.n, load_8_4.p);
      connect(secondary_8_5.n, load_8_5.p);
      connect(secondary_8_6.n, load_8_6.p);
      connect(secondary_8_7.n, load_8_7.p);
      connect(secondary_8_8.n, load_8_8.p);
      connect(secondary_8_9.n, load_8_9.p);
      connect(secondary_8_10.n, load_8_10.p);
      connect(secondary_8_11.n, load_8_11.p);
      connect(secondary_8_12.n, load_8_12.p);
      connect(secondary_8_13.n, load_8_13.p);
      connect(secondary_8_14.n, load_8_14.p);
      connect(primary_9.n, secondary_9_1.p);
      connect(secondary_9_1.n, secondary_9_2.p);
      connect(secondary_9_2.n, secondary_9_3.p);
      connect(secondary_9_3.n, secondary_9_4.p);
      connect(secondary_9_4.n, secondary_9_5.p);
      connect(secondary_9_5.n, secondary_9_6.p);
      connect(secondary_9_6.n, secondary_9_7.p);
      connect(secondary_9_7.n, secondary_9_8.p);
      connect(secondary_9_8.n, secondary_9_9.p);
      connect(secondary_9_9.n, secondary_9_10.p);
      connect(secondary_9_10.n, secondary_9_11.p);
      connect(secondary_9_11.n, secondary_9_12.p);
      connect(secondary_9_12.n, secondary_9_13.p);
      connect(secondary_9_13.n, secondary_9_14.p);
      connect(secondary_9_1.n, load_9_1.p);
      connect(secondary_9_2.n, load_9_2.p);
      connect(secondary_9_3.n, load_9_3.p);
      connect(secondary_9_4.n, load_9_4.p);
      connect(secondary_9_5.n, load_9_5.p);
      connect(secondary_9_6.n, load_9_6.p);
      connect(secondary_9_7.n, load_9_7.p);
      connect(secondary_9_8.n, load_9_8.p);
      connect(secondary_9_9.n, load_9_9.p);
      connect(secondary_9_10.n, load_9_10.p);
      connect(secondary_9_11.n, load_9_11.p);
      connect(secondary_9_12.n, load_9_12.p);
      connect(secondary_9_13.n, load_9_13.p);
      connect(secondary_9_14.n, load_9_14.p);
      connect(primary_10.n, secondary_10_1.p);
      connect(secondary_10_1.n, secondary_10_2.p);
      connect(secondary_10_2.n, secondary_10_3.p);
      connect(secondary_10_3.n, secondary_10_4.p);
      connect(secondary_10_4.n, secondary_10_5.p);
      connect(secondary_10_5.n, secondary_10_6.p);
      connect(secondary_10_6.n, secondary_10_7.p);
      connect(secondary_10_7.n, secondary_10_8.p);
      connect(secondary_10_8.n, secondary_10_9.p);
      connect(secondary_10_9.n, secondary_10_10.p);
      connect(secondary_10_10.n, secondary_10_11.p);
      connect(secondary_10_11.n, secondary_10_12.p);
      connect(secondary_10_12.n, secondary_10_13.p);
      connect(secondary_10_13.n, secondary_10_14.p);
      connect(secondary_10_1.n, load_10_1.p);
      connect(secondary_10_2.n, load_10_2.p);
      connect(secondary_10_3.n, load_10_3.p);
      connect(secondary_10_4.n, load_10_4.p);
      connect(secondary_10_5.n, load_10_5.p);
      connect(secondary_10_6.n, load_10_6.p);
      connect(secondary_10_7.n, load_10_7.p);
      connect(secondary_10_8.n, load_10_8.p);
      connect(secondary_10_9.n, load_10_9.p);
      connect(secondary_10_10.n, load_10_10.p);
      connect(secondary_10_11.n, load_10_11.p);
      connect(secondary_10_12.n, load_10_12.p);
      connect(secondary_10_13.n, load_10_13.p);
      connect(secondary_10_14.n, load_10_14.p);
      connect(primary_11.n, secondary_11_1.p);
      connect(secondary_11_1.n, secondary_11_2.p);
      connect(secondary_11_2.n, secondary_11_3.p);
      connect(secondary_11_3.n, secondary_11_4.p);
      connect(secondary_11_4.n, secondary_11_5.p);
      connect(secondary_11_5.n, secondary_11_6.p);
      connect(secondary_11_6.n, secondary_11_7.p);
      connect(secondary_11_7.n, secondary_11_8.p);
      connect(secondary_11_8.n, secondary_11_9.p);
      connect(secondary_11_9.n, secondary_11_10.p);
      connect(secondary_11_10.n, secondary_11_11.p);
      connect(secondary_11_11.n, secondary_11_12.p);
      connect(secondary_11_12.n, secondary_11_13.p);
      connect(secondary_11_13.n, secondary_11_14.p);
      connect(secondary_11_1.n, load_11_1.p);
      connect(secondary_11_2.n, load_11_2.p);
      connect(secondary_11_3.n, load_11_3.p);
      connect(secondary_11_4.n, load_11_4.p);
      connect(secondary_11_5.n, load_11_5.p);
      connect(secondary_11_6.n, load_11_6.p);
      connect(secondary_11_7.n, load_11_7.p);
      connect(secondary_11_8.n, load_11_8.p);
      connect(secondary_11_9.n, load_11_9.p);
      connect(secondary_11_10.n, load_11_10.p);
      connect(secondary_11_11.n, load_11_11.p);
      connect(secondary_11_12.n, load_11_12.p);
      connect(secondary_11_13.n, load_11_13.p);
      connect(secondary_11_14.n, load_11_14.p);
      connect(primary_12.n, secondary_12_1.p);
      connect(secondary_12_1.n, secondary_12_2.p);
      connect(secondary_12_2.n, secondary_12_3.p);
      connect(secondary_12_3.n, secondary_12_4.p);
      connect(secondary_12_4.n, secondary_12_5.p);
      connect(secondary_12_5.n, secondary_12_6.p);
      connect(secondary_12_6.n, secondary_12_7.p);
      connect(secondary_12_7.n, secondary_12_8.p);
      connect(secondary_12_8.n, secondary_12_9.p);
      connect(secondary_12_9.n, secondary_12_10.p);
      connect(secondary_12_10.n, secondary_12_11.p);
      connect(secondary_12_11.n, secondary_12_12.p);
      connect(secondary_12_12.n, secondary_12_13.p);
      connect(secondary_12_13.n, secondary_12_14.p);
      connect(secondary_12_1.n, load_12_1.p);
      connect(secondary_12_2.n, load_12_2.p);
      connect(secondary_12_3.n, load_12_3.p);
      connect(secondary_12_4.n, load_12_4.p);
      connect(secondary_12_5.n, load_12_5.p);
      connect(secondary_12_6.n, load_12_6.p);
      connect(secondary_12_7.n, load_12_7.p);
      connect(secondary_12_8.n, load_12_8.p);
      connect(secondary_12_9.n, load_12_9.p);
      connect(secondary_12_10.n, load_12_10.p);
      connect(secondary_12_11.n, load_12_11.p);
      connect(secondary_12_12.n, load_12_12.p);
      connect(secondary_12_13.n, load_12_13.p);
      connect(secondary_12_14.n, load_12_14.p);
      connect(primary_13.n, secondary_13_1.p);
      connect(secondary_13_1.n, secondary_13_2.p);
      connect(secondary_13_2.n, secondary_13_3.p);
      connect(secondary_13_3.n, secondary_13_4.p);
      connect(secondary_13_4.n, secondary_13_5.p);
      connect(secondary_13_5.n, secondary_13_6.p);
      connect(secondary_13_6.n, secondary_13_7.p);
      connect(secondary_13_7.n, secondary_13_8.p);
      connect(secondary_13_8.n, secondary_13_9.p);
      connect(secondary_13_9.n, secondary_13_10.p);
      connect(secondary_13_10.n, secondary_13_11.p);
      connect(secondary_13_11.n, secondary_13_12.p);
      connect(secondary_13_12.n, secondary_13_13.p);
      connect(secondary_13_13.n, secondary_13_14.p);
      connect(secondary_13_1.n, load_13_1.p);
      connect(secondary_13_2.n, load_13_2.p);
      connect(secondary_13_3.n, load_13_3.p);
      connect(secondary_13_4.n, load_13_4.p);
      connect(secondary_13_5.n, load_13_5.p);
      connect(secondary_13_6.n, load_13_6.p);
      connect(secondary_13_7.n, load_13_7.p);
      connect(secondary_13_8.n, load_13_8.p);
      connect(secondary_13_9.n, load_13_9.p);
      connect(secondary_13_10.n, load_13_10.p);
      connect(secondary_13_11.n, load_13_11.p);
      connect(secondary_13_12.n, load_13_12.p);
      connect(secondary_13_13.n, load_13_13.p);
      connect(secondary_13_14.n, load_13_14.p);
      connect(primary_14.n, secondary_14_1.p);
      connect(secondary_14_1.n, secondary_14_2.p);
      connect(secondary_14_2.n, secondary_14_3.p);
      connect(secondary_14_3.n, secondary_14_4.p);
      connect(secondary_14_4.n, secondary_14_5.p);
      connect(secondary_14_5.n, secondary_14_6.p);
      connect(secondary_14_6.n, secondary_14_7.p);
      connect(secondary_14_7.n, secondary_14_8.p);
      connect(secondary_14_8.n, secondary_14_9.p);
      connect(secondary_14_9.n, secondary_14_10.p);
      connect(secondary_14_10.n, secondary_14_11.p);
      connect(secondary_14_11.n, secondary_14_12.p);
      connect(secondary_14_12.n, secondary_14_13.p);
      connect(secondary_14_13.n, secondary_14_14.p);
      connect(secondary_14_1.n, load_14_1.p);
      connect(secondary_14_2.n, load_14_2.p);
      connect(secondary_14_3.n, load_14_3.p);
      connect(secondary_14_4.n, load_14_4.p);
      connect(secondary_14_5.n, load_14_5.p);
      connect(secondary_14_6.n, load_14_6.p);
      connect(secondary_14_7.n, load_14_7.p);
      connect(secondary_14_8.n, load_14_8.p);
      connect(secondary_14_9.n, load_14_9.p);
      connect(secondary_14_10.n, load_14_10.p);
      connect(secondary_14_11.n, load_14_11.p);
      connect(secondary_14_12.n, load_14_12.p);
      connect(secondary_14_13.n, load_14_13.p);
      connect(secondary_14_14.n, load_14_14.p);
      annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinearIndividual_N_14_M_14;

    model DistributionSystemLinearIndividual_N_20_M_20
      parameter Integer N = 20 "Number of segments of the primary distribution line";
      parameter Integer M = 20 "Number of segments of each secondary distribution line";
      parameter Real alpha = 10 "Distribution line oversizing factor";
      parameter Real beta = 2 "Ratio between line inductance and line resistance";
      parameter Modelica.SIunits.Resistance R_l = 100 "Resistance of a single load";
      parameter Modelica.SIunits.Resistance R_d2 = R_l / (M ^ 2 * alpha) "Resistance of a secondary distribution segment";
      parameter Modelica.SIunits.Resistance R_d1 = R_l / (M ^ 2 * N ^ 2 * alpha) "Resistance of a primary distribution segment";
      parameter Modelica.SIunits.Voltage V_ref = 600 "Reference source voltage";
      Models.Internals.Impedance primary_1(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_1_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_2(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_2_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_3(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_3_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_4(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_4_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_5(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_5_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_6(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_6_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_7(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_7_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_8(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_8_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_9(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_9_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_10(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_10_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_11(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_11_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_12(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_12_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_13(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_13_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_14(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_14_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_15(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_15_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_16(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_16_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_17(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_17_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_18(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_18_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_19(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_19_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_20(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_20_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Ground sourceGround "Source ground";
      Models.Internals.VoltageSource V_source(V = V_ref) "Voltage source";
    equation
      connect(primary_1.p, V_source.p);
      connect(sourceGround.p, V_source.n);
      connect(primary_1.n, primary_2.p);
      connect(primary_2.n, primary_3.p);
      connect(primary_3.n, primary_4.p);
      connect(primary_4.n, primary_5.p);
      connect(primary_5.n, primary_6.p);
      connect(primary_6.n, primary_7.p);
      connect(primary_7.n, primary_8.p);
      connect(primary_8.n, primary_9.p);
      connect(primary_9.n, primary_10.p);
      connect(primary_10.n, primary_11.p);
      connect(primary_11.n, primary_12.p);
      connect(primary_12.n, primary_13.p);
      connect(primary_13.n, primary_14.p);
      connect(primary_14.n, primary_15.p);
      connect(primary_15.n, primary_16.p);
      connect(primary_16.n, primary_17.p);
      connect(primary_17.n, primary_18.p);
      connect(primary_18.n, primary_19.p);
      connect(primary_19.n, primary_20.p);
      connect(primary_1.n, secondary_1_1.p);
      connect(secondary_1_1.n, secondary_1_2.p);
      connect(secondary_1_2.n, secondary_1_3.p);
      connect(secondary_1_3.n, secondary_1_4.p);
      connect(secondary_1_4.n, secondary_1_5.p);
      connect(secondary_1_5.n, secondary_1_6.p);
      connect(secondary_1_6.n, secondary_1_7.p);
      connect(secondary_1_7.n, secondary_1_8.p);
      connect(secondary_1_8.n, secondary_1_9.p);
      connect(secondary_1_9.n, secondary_1_10.p);
      connect(secondary_1_10.n, secondary_1_11.p);
      connect(secondary_1_11.n, secondary_1_12.p);
      connect(secondary_1_12.n, secondary_1_13.p);
      connect(secondary_1_13.n, secondary_1_14.p);
      connect(secondary_1_14.n, secondary_1_15.p);
      connect(secondary_1_15.n, secondary_1_16.p);
      connect(secondary_1_16.n, secondary_1_17.p);
      connect(secondary_1_17.n, secondary_1_18.p);
      connect(secondary_1_18.n, secondary_1_19.p);
      connect(secondary_1_19.n, secondary_1_20.p);
      connect(secondary_1_1.n, load_1_1.p);
      connect(secondary_1_2.n, load_1_2.p);
      connect(secondary_1_3.n, load_1_3.p);
      connect(secondary_1_4.n, load_1_4.p);
      connect(secondary_1_5.n, load_1_5.p);
      connect(secondary_1_6.n, load_1_6.p);
      connect(secondary_1_7.n, load_1_7.p);
      connect(secondary_1_8.n, load_1_8.p);
      connect(secondary_1_9.n, load_1_9.p);
      connect(secondary_1_10.n, load_1_10.p);
      connect(secondary_1_11.n, load_1_11.p);
      connect(secondary_1_12.n, load_1_12.p);
      connect(secondary_1_13.n, load_1_13.p);
      connect(secondary_1_14.n, load_1_14.p);
      connect(secondary_1_15.n, load_1_15.p);
      connect(secondary_1_16.n, load_1_16.p);
      connect(secondary_1_17.n, load_1_17.p);
      connect(secondary_1_18.n, load_1_18.p);
      connect(secondary_1_19.n, load_1_19.p);
      connect(secondary_1_20.n, load_1_20.p);
      connect(primary_2.n, secondary_2_1.p);
      connect(secondary_2_1.n, secondary_2_2.p);
      connect(secondary_2_2.n, secondary_2_3.p);
      connect(secondary_2_3.n, secondary_2_4.p);
      connect(secondary_2_4.n, secondary_2_5.p);
      connect(secondary_2_5.n, secondary_2_6.p);
      connect(secondary_2_6.n, secondary_2_7.p);
      connect(secondary_2_7.n, secondary_2_8.p);
      connect(secondary_2_8.n, secondary_2_9.p);
      connect(secondary_2_9.n, secondary_2_10.p);
      connect(secondary_2_10.n, secondary_2_11.p);
      connect(secondary_2_11.n, secondary_2_12.p);
      connect(secondary_2_12.n, secondary_2_13.p);
      connect(secondary_2_13.n, secondary_2_14.p);
      connect(secondary_2_14.n, secondary_2_15.p);
      connect(secondary_2_15.n, secondary_2_16.p);
      connect(secondary_2_16.n, secondary_2_17.p);
      connect(secondary_2_17.n, secondary_2_18.p);
      connect(secondary_2_18.n, secondary_2_19.p);
      connect(secondary_2_19.n, secondary_2_20.p);
      connect(secondary_2_1.n, load_2_1.p);
      connect(secondary_2_2.n, load_2_2.p);
      connect(secondary_2_3.n, load_2_3.p);
      connect(secondary_2_4.n, load_2_4.p);
      connect(secondary_2_5.n, load_2_5.p);
      connect(secondary_2_6.n, load_2_6.p);
      connect(secondary_2_7.n, load_2_7.p);
      connect(secondary_2_8.n, load_2_8.p);
      connect(secondary_2_9.n, load_2_9.p);
      connect(secondary_2_10.n, load_2_10.p);
      connect(secondary_2_11.n, load_2_11.p);
      connect(secondary_2_12.n, load_2_12.p);
      connect(secondary_2_13.n, load_2_13.p);
      connect(secondary_2_14.n, load_2_14.p);
      connect(secondary_2_15.n, load_2_15.p);
      connect(secondary_2_16.n, load_2_16.p);
      connect(secondary_2_17.n, load_2_17.p);
      connect(secondary_2_18.n, load_2_18.p);
      connect(secondary_2_19.n, load_2_19.p);
      connect(secondary_2_20.n, load_2_20.p);
      connect(primary_3.n, secondary_3_1.p);
      connect(secondary_3_1.n, secondary_3_2.p);
      connect(secondary_3_2.n, secondary_3_3.p);
      connect(secondary_3_3.n, secondary_3_4.p);
      connect(secondary_3_4.n, secondary_3_5.p);
      connect(secondary_3_5.n, secondary_3_6.p);
      connect(secondary_3_6.n, secondary_3_7.p);
      connect(secondary_3_7.n, secondary_3_8.p);
      connect(secondary_3_8.n, secondary_3_9.p);
      connect(secondary_3_9.n, secondary_3_10.p);
      connect(secondary_3_10.n, secondary_3_11.p);
      connect(secondary_3_11.n, secondary_3_12.p);
      connect(secondary_3_12.n, secondary_3_13.p);
      connect(secondary_3_13.n, secondary_3_14.p);
      connect(secondary_3_14.n, secondary_3_15.p);
      connect(secondary_3_15.n, secondary_3_16.p);
      connect(secondary_3_16.n, secondary_3_17.p);
      connect(secondary_3_17.n, secondary_3_18.p);
      connect(secondary_3_18.n, secondary_3_19.p);
      connect(secondary_3_19.n, secondary_3_20.p);
      connect(secondary_3_1.n, load_3_1.p);
      connect(secondary_3_2.n, load_3_2.p);
      connect(secondary_3_3.n, load_3_3.p);
      connect(secondary_3_4.n, load_3_4.p);
      connect(secondary_3_5.n, load_3_5.p);
      connect(secondary_3_6.n, load_3_6.p);
      connect(secondary_3_7.n, load_3_7.p);
      connect(secondary_3_8.n, load_3_8.p);
      connect(secondary_3_9.n, load_3_9.p);
      connect(secondary_3_10.n, load_3_10.p);
      connect(secondary_3_11.n, load_3_11.p);
      connect(secondary_3_12.n, load_3_12.p);
      connect(secondary_3_13.n, load_3_13.p);
      connect(secondary_3_14.n, load_3_14.p);
      connect(secondary_3_15.n, load_3_15.p);
      connect(secondary_3_16.n, load_3_16.p);
      connect(secondary_3_17.n, load_3_17.p);
      connect(secondary_3_18.n, load_3_18.p);
      connect(secondary_3_19.n, load_3_19.p);
      connect(secondary_3_20.n, load_3_20.p);
      connect(primary_4.n, secondary_4_1.p);
      connect(secondary_4_1.n, secondary_4_2.p);
      connect(secondary_4_2.n, secondary_4_3.p);
      connect(secondary_4_3.n, secondary_4_4.p);
      connect(secondary_4_4.n, secondary_4_5.p);
      connect(secondary_4_5.n, secondary_4_6.p);
      connect(secondary_4_6.n, secondary_4_7.p);
      connect(secondary_4_7.n, secondary_4_8.p);
      connect(secondary_4_8.n, secondary_4_9.p);
      connect(secondary_4_9.n, secondary_4_10.p);
      connect(secondary_4_10.n, secondary_4_11.p);
      connect(secondary_4_11.n, secondary_4_12.p);
      connect(secondary_4_12.n, secondary_4_13.p);
      connect(secondary_4_13.n, secondary_4_14.p);
      connect(secondary_4_14.n, secondary_4_15.p);
      connect(secondary_4_15.n, secondary_4_16.p);
      connect(secondary_4_16.n, secondary_4_17.p);
      connect(secondary_4_17.n, secondary_4_18.p);
      connect(secondary_4_18.n, secondary_4_19.p);
      connect(secondary_4_19.n, secondary_4_20.p);
      connect(secondary_4_1.n, load_4_1.p);
      connect(secondary_4_2.n, load_4_2.p);
      connect(secondary_4_3.n, load_4_3.p);
      connect(secondary_4_4.n, load_4_4.p);
      connect(secondary_4_5.n, load_4_5.p);
      connect(secondary_4_6.n, load_4_6.p);
      connect(secondary_4_7.n, load_4_7.p);
      connect(secondary_4_8.n, load_4_8.p);
      connect(secondary_4_9.n, load_4_9.p);
      connect(secondary_4_10.n, load_4_10.p);
      connect(secondary_4_11.n, load_4_11.p);
      connect(secondary_4_12.n, load_4_12.p);
      connect(secondary_4_13.n, load_4_13.p);
      connect(secondary_4_14.n, load_4_14.p);
      connect(secondary_4_15.n, load_4_15.p);
      connect(secondary_4_16.n, load_4_16.p);
      connect(secondary_4_17.n, load_4_17.p);
      connect(secondary_4_18.n, load_4_18.p);
      connect(secondary_4_19.n, load_4_19.p);
      connect(secondary_4_20.n, load_4_20.p);
      connect(primary_5.n, secondary_5_1.p);
      connect(secondary_5_1.n, secondary_5_2.p);
      connect(secondary_5_2.n, secondary_5_3.p);
      connect(secondary_5_3.n, secondary_5_4.p);
      connect(secondary_5_4.n, secondary_5_5.p);
      connect(secondary_5_5.n, secondary_5_6.p);
      connect(secondary_5_6.n, secondary_5_7.p);
      connect(secondary_5_7.n, secondary_5_8.p);
      connect(secondary_5_8.n, secondary_5_9.p);
      connect(secondary_5_9.n, secondary_5_10.p);
      connect(secondary_5_10.n, secondary_5_11.p);
      connect(secondary_5_11.n, secondary_5_12.p);
      connect(secondary_5_12.n, secondary_5_13.p);
      connect(secondary_5_13.n, secondary_5_14.p);
      connect(secondary_5_14.n, secondary_5_15.p);
      connect(secondary_5_15.n, secondary_5_16.p);
      connect(secondary_5_16.n, secondary_5_17.p);
      connect(secondary_5_17.n, secondary_5_18.p);
      connect(secondary_5_18.n, secondary_5_19.p);
      connect(secondary_5_19.n, secondary_5_20.p);
      connect(secondary_5_1.n, load_5_1.p);
      connect(secondary_5_2.n, load_5_2.p);
      connect(secondary_5_3.n, load_5_3.p);
      connect(secondary_5_4.n, load_5_4.p);
      connect(secondary_5_5.n, load_5_5.p);
      connect(secondary_5_6.n, load_5_6.p);
      connect(secondary_5_7.n, load_5_7.p);
      connect(secondary_5_8.n, load_5_8.p);
      connect(secondary_5_9.n, load_5_9.p);
      connect(secondary_5_10.n, load_5_10.p);
      connect(secondary_5_11.n, load_5_11.p);
      connect(secondary_5_12.n, load_5_12.p);
      connect(secondary_5_13.n, load_5_13.p);
      connect(secondary_5_14.n, load_5_14.p);
      connect(secondary_5_15.n, load_5_15.p);
      connect(secondary_5_16.n, load_5_16.p);
      connect(secondary_5_17.n, load_5_17.p);
      connect(secondary_5_18.n, load_5_18.p);
      connect(secondary_5_19.n, load_5_19.p);
      connect(secondary_5_20.n, load_5_20.p);
      connect(primary_6.n, secondary_6_1.p);
      connect(secondary_6_1.n, secondary_6_2.p);
      connect(secondary_6_2.n, secondary_6_3.p);
      connect(secondary_6_3.n, secondary_6_4.p);
      connect(secondary_6_4.n, secondary_6_5.p);
      connect(secondary_6_5.n, secondary_6_6.p);
      connect(secondary_6_6.n, secondary_6_7.p);
      connect(secondary_6_7.n, secondary_6_8.p);
      connect(secondary_6_8.n, secondary_6_9.p);
      connect(secondary_6_9.n, secondary_6_10.p);
      connect(secondary_6_10.n, secondary_6_11.p);
      connect(secondary_6_11.n, secondary_6_12.p);
      connect(secondary_6_12.n, secondary_6_13.p);
      connect(secondary_6_13.n, secondary_6_14.p);
      connect(secondary_6_14.n, secondary_6_15.p);
      connect(secondary_6_15.n, secondary_6_16.p);
      connect(secondary_6_16.n, secondary_6_17.p);
      connect(secondary_6_17.n, secondary_6_18.p);
      connect(secondary_6_18.n, secondary_6_19.p);
      connect(secondary_6_19.n, secondary_6_20.p);
      connect(secondary_6_1.n, load_6_1.p);
      connect(secondary_6_2.n, load_6_2.p);
      connect(secondary_6_3.n, load_6_3.p);
      connect(secondary_6_4.n, load_6_4.p);
      connect(secondary_6_5.n, load_6_5.p);
      connect(secondary_6_6.n, load_6_6.p);
      connect(secondary_6_7.n, load_6_7.p);
      connect(secondary_6_8.n, load_6_8.p);
      connect(secondary_6_9.n, load_6_9.p);
      connect(secondary_6_10.n, load_6_10.p);
      connect(secondary_6_11.n, load_6_11.p);
      connect(secondary_6_12.n, load_6_12.p);
      connect(secondary_6_13.n, load_6_13.p);
      connect(secondary_6_14.n, load_6_14.p);
      connect(secondary_6_15.n, load_6_15.p);
      connect(secondary_6_16.n, load_6_16.p);
      connect(secondary_6_17.n, load_6_17.p);
      connect(secondary_6_18.n, load_6_18.p);
      connect(secondary_6_19.n, load_6_19.p);
      connect(secondary_6_20.n, load_6_20.p);
      connect(primary_7.n, secondary_7_1.p);
      connect(secondary_7_1.n, secondary_7_2.p);
      connect(secondary_7_2.n, secondary_7_3.p);
      connect(secondary_7_3.n, secondary_7_4.p);
      connect(secondary_7_4.n, secondary_7_5.p);
      connect(secondary_7_5.n, secondary_7_6.p);
      connect(secondary_7_6.n, secondary_7_7.p);
      connect(secondary_7_7.n, secondary_7_8.p);
      connect(secondary_7_8.n, secondary_7_9.p);
      connect(secondary_7_9.n, secondary_7_10.p);
      connect(secondary_7_10.n, secondary_7_11.p);
      connect(secondary_7_11.n, secondary_7_12.p);
      connect(secondary_7_12.n, secondary_7_13.p);
      connect(secondary_7_13.n, secondary_7_14.p);
      connect(secondary_7_14.n, secondary_7_15.p);
      connect(secondary_7_15.n, secondary_7_16.p);
      connect(secondary_7_16.n, secondary_7_17.p);
      connect(secondary_7_17.n, secondary_7_18.p);
      connect(secondary_7_18.n, secondary_7_19.p);
      connect(secondary_7_19.n, secondary_7_20.p);
      connect(secondary_7_1.n, load_7_1.p);
      connect(secondary_7_2.n, load_7_2.p);
      connect(secondary_7_3.n, load_7_3.p);
      connect(secondary_7_4.n, load_7_4.p);
      connect(secondary_7_5.n, load_7_5.p);
      connect(secondary_7_6.n, load_7_6.p);
      connect(secondary_7_7.n, load_7_7.p);
      connect(secondary_7_8.n, load_7_8.p);
      connect(secondary_7_9.n, load_7_9.p);
      connect(secondary_7_10.n, load_7_10.p);
      connect(secondary_7_11.n, load_7_11.p);
      connect(secondary_7_12.n, load_7_12.p);
      connect(secondary_7_13.n, load_7_13.p);
      connect(secondary_7_14.n, load_7_14.p);
      connect(secondary_7_15.n, load_7_15.p);
      connect(secondary_7_16.n, load_7_16.p);
      connect(secondary_7_17.n, load_7_17.p);
      connect(secondary_7_18.n, load_7_18.p);
      connect(secondary_7_19.n, load_7_19.p);
      connect(secondary_7_20.n, load_7_20.p);
      connect(primary_8.n, secondary_8_1.p);
      connect(secondary_8_1.n, secondary_8_2.p);
      connect(secondary_8_2.n, secondary_8_3.p);
      connect(secondary_8_3.n, secondary_8_4.p);
      connect(secondary_8_4.n, secondary_8_5.p);
      connect(secondary_8_5.n, secondary_8_6.p);
      connect(secondary_8_6.n, secondary_8_7.p);
      connect(secondary_8_7.n, secondary_8_8.p);
      connect(secondary_8_8.n, secondary_8_9.p);
      connect(secondary_8_9.n, secondary_8_10.p);
      connect(secondary_8_10.n, secondary_8_11.p);
      connect(secondary_8_11.n, secondary_8_12.p);
      connect(secondary_8_12.n, secondary_8_13.p);
      connect(secondary_8_13.n, secondary_8_14.p);
      connect(secondary_8_14.n, secondary_8_15.p);
      connect(secondary_8_15.n, secondary_8_16.p);
      connect(secondary_8_16.n, secondary_8_17.p);
      connect(secondary_8_17.n, secondary_8_18.p);
      connect(secondary_8_18.n, secondary_8_19.p);
      connect(secondary_8_19.n, secondary_8_20.p);
      connect(secondary_8_1.n, load_8_1.p);
      connect(secondary_8_2.n, load_8_2.p);
      connect(secondary_8_3.n, load_8_3.p);
      connect(secondary_8_4.n, load_8_4.p);
      connect(secondary_8_5.n, load_8_5.p);
      connect(secondary_8_6.n, load_8_6.p);
      connect(secondary_8_7.n, load_8_7.p);
      connect(secondary_8_8.n, load_8_8.p);
      connect(secondary_8_9.n, load_8_9.p);
      connect(secondary_8_10.n, load_8_10.p);
      connect(secondary_8_11.n, load_8_11.p);
      connect(secondary_8_12.n, load_8_12.p);
      connect(secondary_8_13.n, load_8_13.p);
      connect(secondary_8_14.n, load_8_14.p);
      connect(secondary_8_15.n, load_8_15.p);
      connect(secondary_8_16.n, load_8_16.p);
      connect(secondary_8_17.n, load_8_17.p);
      connect(secondary_8_18.n, load_8_18.p);
      connect(secondary_8_19.n, load_8_19.p);
      connect(secondary_8_20.n, load_8_20.p);
      connect(primary_9.n, secondary_9_1.p);
      connect(secondary_9_1.n, secondary_9_2.p);
      connect(secondary_9_2.n, secondary_9_3.p);
      connect(secondary_9_3.n, secondary_9_4.p);
      connect(secondary_9_4.n, secondary_9_5.p);
      connect(secondary_9_5.n, secondary_9_6.p);
      connect(secondary_9_6.n, secondary_9_7.p);
      connect(secondary_9_7.n, secondary_9_8.p);
      connect(secondary_9_8.n, secondary_9_9.p);
      connect(secondary_9_9.n, secondary_9_10.p);
      connect(secondary_9_10.n, secondary_9_11.p);
      connect(secondary_9_11.n, secondary_9_12.p);
      connect(secondary_9_12.n, secondary_9_13.p);
      connect(secondary_9_13.n, secondary_9_14.p);
      connect(secondary_9_14.n, secondary_9_15.p);
      connect(secondary_9_15.n, secondary_9_16.p);
      connect(secondary_9_16.n, secondary_9_17.p);
      connect(secondary_9_17.n, secondary_9_18.p);
      connect(secondary_9_18.n, secondary_9_19.p);
      connect(secondary_9_19.n, secondary_9_20.p);
      connect(secondary_9_1.n, load_9_1.p);
      connect(secondary_9_2.n, load_9_2.p);
      connect(secondary_9_3.n, load_9_3.p);
      connect(secondary_9_4.n, load_9_4.p);
      connect(secondary_9_5.n, load_9_5.p);
      connect(secondary_9_6.n, load_9_6.p);
      connect(secondary_9_7.n, load_9_7.p);
      connect(secondary_9_8.n, load_9_8.p);
      connect(secondary_9_9.n, load_9_9.p);
      connect(secondary_9_10.n, load_9_10.p);
      connect(secondary_9_11.n, load_9_11.p);
      connect(secondary_9_12.n, load_9_12.p);
      connect(secondary_9_13.n, load_9_13.p);
      connect(secondary_9_14.n, load_9_14.p);
      connect(secondary_9_15.n, load_9_15.p);
      connect(secondary_9_16.n, load_9_16.p);
      connect(secondary_9_17.n, load_9_17.p);
      connect(secondary_9_18.n, load_9_18.p);
      connect(secondary_9_19.n, load_9_19.p);
      connect(secondary_9_20.n, load_9_20.p);
      connect(primary_10.n, secondary_10_1.p);
      connect(secondary_10_1.n, secondary_10_2.p);
      connect(secondary_10_2.n, secondary_10_3.p);
      connect(secondary_10_3.n, secondary_10_4.p);
      connect(secondary_10_4.n, secondary_10_5.p);
      connect(secondary_10_5.n, secondary_10_6.p);
      connect(secondary_10_6.n, secondary_10_7.p);
      connect(secondary_10_7.n, secondary_10_8.p);
      connect(secondary_10_8.n, secondary_10_9.p);
      connect(secondary_10_9.n, secondary_10_10.p);
      connect(secondary_10_10.n, secondary_10_11.p);
      connect(secondary_10_11.n, secondary_10_12.p);
      connect(secondary_10_12.n, secondary_10_13.p);
      connect(secondary_10_13.n, secondary_10_14.p);
      connect(secondary_10_14.n, secondary_10_15.p);
      connect(secondary_10_15.n, secondary_10_16.p);
      connect(secondary_10_16.n, secondary_10_17.p);
      connect(secondary_10_17.n, secondary_10_18.p);
      connect(secondary_10_18.n, secondary_10_19.p);
      connect(secondary_10_19.n, secondary_10_20.p);
      connect(secondary_10_1.n, load_10_1.p);
      connect(secondary_10_2.n, load_10_2.p);
      connect(secondary_10_3.n, load_10_3.p);
      connect(secondary_10_4.n, load_10_4.p);
      connect(secondary_10_5.n, load_10_5.p);
      connect(secondary_10_6.n, load_10_6.p);
      connect(secondary_10_7.n, load_10_7.p);
      connect(secondary_10_8.n, load_10_8.p);
      connect(secondary_10_9.n, load_10_9.p);
      connect(secondary_10_10.n, load_10_10.p);
      connect(secondary_10_11.n, load_10_11.p);
      connect(secondary_10_12.n, load_10_12.p);
      connect(secondary_10_13.n, load_10_13.p);
      connect(secondary_10_14.n, load_10_14.p);
      connect(secondary_10_15.n, load_10_15.p);
      connect(secondary_10_16.n, load_10_16.p);
      connect(secondary_10_17.n, load_10_17.p);
      connect(secondary_10_18.n, load_10_18.p);
      connect(secondary_10_19.n, load_10_19.p);
      connect(secondary_10_20.n, load_10_20.p);
      connect(primary_11.n, secondary_11_1.p);
      connect(secondary_11_1.n, secondary_11_2.p);
      connect(secondary_11_2.n, secondary_11_3.p);
      connect(secondary_11_3.n, secondary_11_4.p);
      connect(secondary_11_4.n, secondary_11_5.p);
      connect(secondary_11_5.n, secondary_11_6.p);
      connect(secondary_11_6.n, secondary_11_7.p);
      connect(secondary_11_7.n, secondary_11_8.p);
      connect(secondary_11_8.n, secondary_11_9.p);
      connect(secondary_11_9.n, secondary_11_10.p);
      connect(secondary_11_10.n, secondary_11_11.p);
      connect(secondary_11_11.n, secondary_11_12.p);
      connect(secondary_11_12.n, secondary_11_13.p);
      connect(secondary_11_13.n, secondary_11_14.p);
      connect(secondary_11_14.n, secondary_11_15.p);
      connect(secondary_11_15.n, secondary_11_16.p);
      connect(secondary_11_16.n, secondary_11_17.p);
      connect(secondary_11_17.n, secondary_11_18.p);
      connect(secondary_11_18.n, secondary_11_19.p);
      connect(secondary_11_19.n, secondary_11_20.p);
      connect(secondary_11_1.n, load_11_1.p);
      connect(secondary_11_2.n, load_11_2.p);
      connect(secondary_11_3.n, load_11_3.p);
      connect(secondary_11_4.n, load_11_4.p);
      connect(secondary_11_5.n, load_11_5.p);
      connect(secondary_11_6.n, load_11_6.p);
      connect(secondary_11_7.n, load_11_7.p);
      connect(secondary_11_8.n, load_11_8.p);
      connect(secondary_11_9.n, load_11_9.p);
      connect(secondary_11_10.n, load_11_10.p);
      connect(secondary_11_11.n, load_11_11.p);
      connect(secondary_11_12.n, load_11_12.p);
      connect(secondary_11_13.n, load_11_13.p);
      connect(secondary_11_14.n, load_11_14.p);
      connect(secondary_11_15.n, load_11_15.p);
      connect(secondary_11_16.n, load_11_16.p);
      connect(secondary_11_17.n, load_11_17.p);
      connect(secondary_11_18.n, load_11_18.p);
      connect(secondary_11_19.n, load_11_19.p);
      connect(secondary_11_20.n, load_11_20.p);
      connect(primary_12.n, secondary_12_1.p);
      connect(secondary_12_1.n, secondary_12_2.p);
      connect(secondary_12_2.n, secondary_12_3.p);
      connect(secondary_12_3.n, secondary_12_4.p);
      connect(secondary_12_4.n, secondary_12_5.p);
      connect(secondary_12_5.n, secondary_12_6.p);
      connect(secondary_12_6.n, secondary_12_7.p);
      connect(secondary_12_7.n, secondary_12_8.p);
      connect(secondary_12_8.n, secondary_12_9.p);
      connect(secondary_12_9.n, secondary_12_10.p);
      connect(secondary_12_10.n, secondary_12_11.p);
      connect(secondary_12_11.n, secondary_12_12.p);
      connect(secondary_12_12.n, secondary_12_13.p);
      connect(secondary_12_13.n, secondary_12_14.p);
      connect(secondary_12_14.n, secondary_12_15.p);
      connect(secondary_12_15.n, secondary_12_16.p);
      connect(secondary_12_16.n, secondary_12_17.p);
      connect(secondary_12_17.n, secondary_12_18.p);
      connect(secondary_12_18.n, secondary_12_19.p);
      connect(secondary_12_19.n, secondary_12_20.p);
      connect(secondary_12_1.n, load_12_1.p);
      connect(secondary_12_2.n, load_12_2.p);
      connect(secondary_12_3.n, load_12_3.p);
      connect(secondary_12_4.n, load_12_4.p);
      connect(secondary_12_5.n, load_12_5.p);
      connect(secondary_12_6.n, load_12_6.p);
      connect(secondary_12_7.n, load_12_7.p);
      connect(secondary_12_8.n, load_12_8.p);
      connect(secondary_12_9.n, load_12_9.p);
      connect(secondary_12_10.n, load_12_10.p);
      connect(secondary_12_11.n, load_12_11.p);
      connect(secondary_12_12.n, load_12_12.p);
      connect(secondary_12_13.n, load_12_13.p);
      connect(secondary_12_14.n, load_12_14.p);
      connect(secondary_12_15.n, load_12_15.p);
      connect(secondary_12_16.n, load_12_16.p);
      connect(secondary_12_17.n, load_12_17.p);
      connect(secondary_12_18.n, load_12_18.p);
      connect(secondary_12_19.n, load_12_19.p);
      connect(secondary_12_20.n, load_12_20.p);
      connect(primary_13.n, secondary_13_1.p);
      connect(secondary_13_1.n, secondary_13_2.p);
      connect(secondary_13_2.n, secondary_13_3.p);
      connect(secondary_13_3.n, secondary_13_4.p);
      connect(secondary_13_4.n, secondary_13_5.p);
      connect(secondary_13_5.n, secondary_13_6.p);
      connect(secondary_13_6.n, secondary_13_7.p);
      connect(secondary_13_7.n, secondary_13_8.p);
      connect(secondary_13_8.n, secondary_13_9.p);
      connect(secondary_13_9.n, secondary_13_10.p);
      connect(secondary_13_10.n, secondary_13_11.p);
      connect(secondary_13_11.n, secondary_13_12.p);
      connect(secondary_13_12.n, secondary_13_13.p);
      connect(secondary_13_13.n, secondary_13_14.p);
      connect(secondary_13_14.n, secondary_13_15.p);
      connect(secondary_13_15.n, secondary_13_16.p);
      connect(secondary_13_16.n, secondary_13_17.p);
      connect(secondary_13_17.n, secondary_13_18.p);
      connect(secondary_13_18.n, secondary_13_19.p);
      connect(secondary_13_19.n, secondary_13_20.p);
      connect(secondary_13_1.n, load_13_1.p);
      connect(secondary_13_2.n, load_13_2.p);
      connect(secondary_13_3.n, load_13_3.p);
      connect(secondary_13_4.n, load_13_4.p);
      connect(secondary_13_5.n, load_13_5.p);
      connect(secondary_13_6.n, load_13_6.p);
      connect(secondary_13_7.n, load_13_7.p);
      connect(secondary_13_8.n, load_13_8.p);
      connect(secondary_13_9.n, load_13_9.p);
      connect(secondary_13_10.n, load_13_10.p);
      connect(secondary_13_11.n, load_13_11.p);
      connect(secondary_13_12.n, load_13_12.p);
      connect(secondary_13_13.n, load_13_13.p);
      connect(secondary_13_14.n, load_13_14.p);
      connect(secondary_13_15.n, load_13_15.p);
      connect(secondary_13_16.n, load_13_16.p);
      connect(secondary_13_17.n, load_13_17.p);
      connect(secondary_13_18.n, load_13_18.p);
      connect(secondary_13_19.n, load_13_19.p);
      connect(secondary_13_20.n, load_13_20.p);
      connect(primary_14.n, secondary_14_1.p);
      connect(secondary_14_1.n, secondary_14_2.p);
      connect(secondary_14_2.n, secondary_14_3.p);
      connect(secondary_14_3.n, secondary_14_4.p);
      connect(secondary_14_4.n, secondary_14_5.p);
      connect(secondary_14_5.n, secondary_14_6.p);
      connect(secondary_14_6.n, secondary_14_7.p);
      connect(secondary_14_7.n, secondary_14_8.p);
      connect(secondary_14_8.n, secondary_14_9.p);
      connect(secondary_14_9.n, secondary_14_10.p);
      connect(secondary_14_10.n, secondary_14_11.p);
      connect(secondary_14_11.n, secondary_14_12.p);
      connect(secondary_14_12.n, secondary_14_13.p);
      connect(secondary_14_13.n, secondary_14_14.p);
      connect(secondary_14_14.n, secondary_14_15.p);
      connect(secondary_14_15.n, secondary_14_16.p);
      connect(secondary_14_16.n, secondary_14_17.p);
      connect(secondary_14_17.n, secondary_14_18.p);
      connect(secondary_14_18.n, secondary_14_19.p);
      connect(secondary_14_19.n, secondary_14_20.p);
      connect(secondary_14_1.n, load_14_1.p);
      connect(secondary_14_2.n, load_14_2.p);
      connect(secondary_14_3.n, load_14_3.p);
      connect(secondary_14_4.n, load_14_4.p);
      connect(secondary_14_5.n, load_14_5.p);
      connect(secondary_14_6.n, load_14_6.p);
      connect(secondary_14_7.n, load_14_7.p);
      connect(secondary_14_8.n, load_14_8.p);
      connect(secondary_14_9.n, load_14_9.p);
      connect(secondary_14_10.n, load_14_10.p);
      connect(secondary_14_11.n, load_14_11.p);
      connect(secondary_14_12.n, load_14_12.p);
      connect(secondary_14_13.n, load_14_13.p);
      connect(secondary_14_14.n, load_14_14.p);
      connect(secondary_14_15.n, load_14_15.p);
      connect(secondary_14_16.n, load_14_16.p);
      connect(secondary_14_17.n, load_14_17.p);
      connect(secondary_14_18.n, load_14_18.p);
      connect(secondary_14_19.n, load_14_19.p);
      connect(secondary_14_20.n, load_14_20.p);
      connect(primary_15.n, secondary_15_1.p);
      connect(secondary_15_1.n, secondary_15_2.p);
      connect(secondary_15_2.n, secondary_15_3.p);
      connect(secondary_15_3.n, secondary_15_4.p);
      connect(secondary_15_4.n, secondary_15_5.p);
      connect(secondary_15_5.n, secondary_15_6.p);
      connect(secondary_15_6.n, secondary_15_7.p);
      connect(secondary_15_7.n, secondary_15_8.p);
      connect(secondary_15_8.n, secondary_15_9.p);
      connect(secondary_15_9.n, secondary_15_10.p);
      connect(secondary_15_10.n, secondary_15_11.p);
      connect(secondary_15_11.n, secondary_15_12.p);
      connect(secondary_15_12.n, secondary_15_13.p);
      connect(secondary_15_13.n, secondary_15_14.p);
      connect(secondary_15_14.n, secondary_15_15.p);
      connect(secondary_15_15.n, secondary_15_16.p);
      connect(secondary_15_16.n, secondary_15_17.p);
      connect(secondary_15_17.n, secondary_15_18.p);
      connect(secondary_15_18.n, secondary_15_19.p);
      connect(secondary_15_19.n, secondary_15_20.p);
      connect(secondary_15_1.n, load_15_1.p);
      connect(secondary_15_2.n, load_15_2.p);
      connect(secondary_15_3.n, load_15_3.p);
      connect(secondary_15_4.n, load_15_4.p);
      connect(secondary_15_5.n, load_15_5.p);
      connect(secondary_15_6.n, load_15_6.p);
      connect(secondary_15_7.n, load_15_7.p);
      connect(secondary_15_8.n, load_15_8.p);
      connect(secondary_15_9.n, load_15_9.p);
      connect(secondary_15_10.n, load_15_10.p);
      connect(secondary_15_11.n, load_15_11.p);
      connect(secondary_15_12.n, load_15_12.p);
      connect(secondary_15_13.n, load_15_13.p);
      connect(secondary_15_14.n, load_15_14.p);
      connect(secondary_15_15.n, load_15_15.p);
      connect(secondary_15_16.n, load_15_16.p);
      connect(secondary_15_17.n, load_15_17.p);
      connect(secondary_15_18.n, load_15_18.p);
      connect(secondary_15_19.n, load_15_19.p);
      connect(secondary_15_20.n, load_15_20.p);
      connect(primary_16.n, secondary_16_1.p);
      connect(secondary_16_1.n, secondary_16_2.p);
      connect(secondary_16_2.n, secondary_16_3.p);
      connect(secondary_16_3.n, secondary_16_4.p);
      connect(secondary_16_4.n, secondary_16_5.p);
      connect(secondary_16_5.n, secondary_16_6.p);
      connect(secondary_16_6.n, secondary_16_7.p);
      connect(secondary_16_7.n, secondary_16_8.p);
      connect(secondary_16_8.n, secondary_16_9.p);
      connect(secondary_16_9.n, secondary_16_10.p);
      connect(secondary_16_10.n, secondary_16_11.p);
      connect(secondary_16_11.n, secondary_16_12.p);
      connect(secondary_16_12.n, secondary_16_13.p);
      connect(secondary_16_13.n, secondary_16_14.p);
      connect(secondary_16_14.n, secondary_16_15.p);
      connect(secondary_16_15.n, secondary_16_16.p);
      connect(secondary_16_16.n, secondary_16_17.p);
      connect(secondary_16_17.n, secondary_16_18.p);
      connect(secondary_16_18.n, secondary_16_19.p);
      connect(secondary_16_19.n, secondary_16_20.p);
      connect(secondary_16_1.n, load_16_1.p);
      connect(secondary_16_2.n, load_16_2.p);
      connect(secondary_16_3.n, load_16_3.p);
      connect(secondary_16_4.n, load_16_4.p);
      connect(secondary_16_5.n, load_16_5.p);
      connect(secondary_16_6.n, load_16_6.p);
      connect(secondary_16_7.n, load_16_7.p);
      connect(secondary_16_8.n, load_16_8.p);
      connect(secondary_16_9.n, load_16_9.p);
      connect(secondary_16_10.n, load_16_10.p);
      connect(secondary_16_11.n, load_16_11.p);
      connect(secondary_16_12.n, load_16_12.p);
      connect(secondary_16_13.n, load_16_13.p);
      connect(secondary_16_14.n, load_16_14.p);
      connect(secondary_16_15.n, load_16_15.p);
      connect(secondary_16_16.n, load_16_16.p);
      connect(secondary_16_17.n, load_16_17.p);
      connect(secondary_16_18.n, load_16_18.p);
      connect(secondary_16_19.n, load_16_19.p);
      connect(secondary_16_20.n, load_16_20.p);
      connect(primary_17.n, secondary_17_1.p);
      connect(secondary_17_1.n, secondary_17_2.p);
      connect(secondary_17_2.n, secondary_17_3.p);
      connect(secondary_17_3.n, secondary_17_4.p);
      connect(secondary_17_4.n, secondary_17_5.p);
      connect(secondary_17_5.n, secondary_17_6.p);
      connect(secondary_17_6.n, secondary_17_7.p);
      connect(secondary_17_7.n, secondary_17_8.p);
      connect(secondary_17_8.n, secondary_17_9.p);
      connect(secondary_17_9.n, secondary_17_10.p);
      connect(secondary_17_10.n, secondary_17_11.p);
      connect(secondary_17_11.n, secondary_17_12.p);
      connect(secondary_17_12.n, secondary_17_13.p);
      connect(secondary_17_13.n, secondary_17_14.p);
      connect(secondary_17_14.n, secondary_17_15.p);
      connect(secondary_17_15.n, secondary_17_16.p);
      connect(secondary_17_16.n, secondary_17_17.p);
      connect(secondary_17_17.n, secondary_17_18.p);
      connect(secondary_17_18.n, secondary_17_19.p);
      connect(secondary_17_19.n, secondary_17_20.p);
      connect(secondary_17_1.n, load_17_1.p);
      connect(secondary_17_2.n, load_17_2.p);
      connect(secondary_17_3.n, load_17_3.p);
      connect(secondary_17_4.n, load_17_4.p);
      connect(secondary_17_5.n, load_17_5.p);
      connect(secondary_17_6.n, load_17_6.p);
      connect(secondary_17_7.n, load_17_7.p);
      connect(secondary_17_8.n, load_17_8.p);
      connect(secondary_17_9.n, load_17_9.p);
      connect(secondary_17_10.n, load_17_10.p);
      connect(secondary_17_11.n, load_17_11.p);
      connect(secondary_17_12.n, load_17_12.p);
      connect(secondary_17_13.n, load_17_13.p);
      connect(secondary_17_14.n, load_17_14.p);
      connect(secondary_17_15.n, load_17_15.p);
      connect(secondary_17_16.n, load_17_16.p);
      connect(secondary_17_17.n, load_17_17.p);
      connect(secondary_17_18.n, load_17_18.p);
      connect(secondary_17_19.n, load_17_19.p);
      connect(secondary_17_20.n, load_17_20.p);
      connect(primary_18.n, secondary_18_1.p);
      connect(secondary_18_1.n, secondary_18_2.p);
      connect(secondary_18_2.n, secondary_18_3.p);
      connect(secondary_18_3.n, secondary_18_4.p);
      connect(secondary_18_4.n, secondary_18_5.p);
      connect(secondary_18_5.n, secondary_18_6.p);
      connect(secondary_18_6.n, secondary_18_7.p);
      connect(secondary_18_7.n, secondary_18_8.p);
      connect(secondary_18_8.n, secondary_18_9.p);
      connect(secondary_18_9.n, secondary_18_10.p);
      connect(secondary_18_10.n, secondary_18_11.p);
      connect(secondary_18_11.n, secondary_18_12.p);
      connect(secondary_18_12.n, secondary_18_13.p);
      connect(secondary_18_13.n, secondary_18_14.p);
      connect(secondary_18_14.n, secondary_18_15.p);
      connect(secondary_18_15.n, secondary_18_16.p);
      connect(secondary_18_16.n, secondary_18_17.p);
      connect(secondary_18_17.n, secondary_18_18.p);
      connect(secondary_18_18.n, secondary_18_19.p);
      connect(secondary_18_19.n, secondary_18_20.p);
      connect(secondary_18_1.n, load_18_1.p);
      connect(secondary_18_2.n, load_18_2.p);
      connect(secondary_18_3.n, load_18_3.p);
      connect(secondary_18_4.n, load_18_4.p);
      connect(secondary_18_5.n, load_18_5.p);
      connect(secondary_18_6.n, load_18_6.p);
      connect(secondary_18_7.n, load_18_7.p);
      connect(secondary_18_8.n, load_18_8.p);
      connect(secondary_18_9.n, load_18_9.p);
      connect(secondary_18_10.n, load_18_10.p);
      connect(secondary_18_11.n, load_18_11.p);
      connect(secondary_18_12.n, load_18_12.p);
      connect(secondary_18_13.n, load_18_13.p);
      connect(secondary_18_14.n, load_18_14.p);
      connect(secondary_18_15.n, load_18_15.p);
      connect(secondary_18_16.n, load_18_16.p);
      connect(secondary_18_17.n, load_18_17.p);
      connect(secondary_18_18.n, load_18_18.p);
      connect(secondary_18_19.n, load_18_19.p);
      connect(secondary_18_20.n, load_18_20.p);
      connect(primary_19.n, secondary_19_1.p);
      connect(secondary_19_1.n, secondary_19_2.p);
      connect(secondary_19_2.n, secondary_19_3.p);
      connect(secondary_19_3.n, secondary_19_4.p);
      connect(secondary_19_4.n, secondary_19_5.p);
      connect(secondary_19_5.n, secondary_19_6.p);
      connect(secondary_19_6.n, secondary_19_7.p);
      connect(secondary_19_7.n, secondary_19_8.p);
      connect(secondary_19_8.n, secondary_19_9.p);
      connect(secondary_19_9.n, secondary_19_10.p);
      connect(secondary_19_10.n, secondary_19_11.p);
      connect(secondary_19_11.n, secondary_19_12.p);
      connect(secondary_19_12.n, secondary_19_13.p);
      connect(secondary_19_13.n, secondary_19_14.p);
      connect(secondary_19_14.n, secondary_19_15.p);
      connect(secondary_19_15.n, secondary_19_16.p);
      connect(secondary_19_16.n, secondary_19_17.p);
      connect(secondary_19_17.n, secondary_19_18.p);
      connect(secondary_19_18.n, secondary_19_19.p);
      connect(secondary_19_19.n, secondary_19_20.p);
      connect(secondary_19_1.n, load_19_1.p);
      connect(secondary_19_2.n, load_19_2.p);
      connect(secondary_19_3.n, load_19_3.p);
      connect(secondary_19_4.n, load_19_4.p);
      connect(secondary_19_5.n, load_19_5.p);
      connect(secondary_19_6.n, load_19_6.p);
      connect(secondary_19_7.n, load_19_7.p);
      connect(secondary_19_8.n, load_19_8.p);
      connect(secondary_19_9.n, load_19_9.p);
      connect(secondary_19_10.n, load_19_10.p);
      connect(secondary_19_11.n, load_19_11.p);
      connect(secondary_19_12.n, load_19_12.p);
      connect(secondary_19_13.n, load_19_13.p);
      connect(secondary_19_14.n, load_19_14.p);
      connect(secondary_19_15.n, load_19_15.p);
      connect(secondary_19_16.n, load_19_16.p);
      connect(secondary_19_17.n, load_19_17.p);
      connect(secondary_19_18.n, load_19_18.p);
      connect(secondary_19_19.n, load_19_19.p);
      connect(secondary_19_20.n, load_19_20.p);
      connect(primary_20.n, secondary_20_1.p);
      connect(secondary_20_1.n, secondary_20_2.p);
      connect(secondary_20_2.n, secondary_20_3.p);
      connect(secondary_20_3.n, secondary_20_4.p);
      connect(secondary_20_4.n, secondary_20_5.p);
      connect(secondary_20_5.n, secondary_20_6.p);
      connect(secondary_20_6.n, secondary_20_7.p);
      connect(secondary_20_7.n, secondary_20_8.p);
      connect(secondary_20_8.n, secondary_20_9.p);
      connect(secondary_20_9.n, secondary_20_10.p);
      connect(secondary_20_10.n, secondary_20_11.p);
      connect(secondary_20_11.n, secondary_20_12.p);
      connect(secondary_20_12.n, secondary_20_13.p);
      connect(secondary_20_13.n, secondary_20_14.p);
      connect(secondary_20_14.n, secondary_20_15.p);
      connect(secondary_20_15.n, secondary_20_16.p);
      connect(secondary_20_16.n, secondary_20_17.p);
      connect(secondary_20_17.n, secondary_20_18.p);
      connect(secondary_20_18.n, secondary_20_19.p);
      connect(secondary_20_19.n, secondary_20_20.p);
      connect(secondary_20_1.n, load_20_1.p);
      connect(secondary_20_2.n, load_20_2.p);
      connect(secondary_20_3.n, load_20_3.p);
      connect(secondary_20_4.n, load_20_4.p);
      connect(secondary_20_5.n, load_20_5.p);
      connect(secondary_20_6.n, load_20_6.p);
      connect(secondary_20_7.n, load_20_7.p);
      connect(secondary_20_8.n, load_20_8.p);
      connect(secondary_20_9.n, load_20_9.p);
      connect(secondary_20_10.n, load_20_10.p);
      connect(secondary_20_11.n, load_20_11.p);
      connect(secondary_20_12.n, load_20_12.p);
      connect(secondary_20_13.n, load_20_13.p);
      connect(secondary_20_14.n, load_20_14.p);
      connect(secondary_20_15.n, load_20_15.p);
      connect(secondary_20_16.n, load_20_16.p);
      connect(secondary_20_17.n, load_20_17.p);
      connect(secondary_20_18.n, load_20_18.p);
      connect(secondary_20_19.n, load_20_19.p);
      connect(secondary_20_20.n, load_20_20.p);
      annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinearIndividual_N_20_M_20;

    model DistributionSystemLinearIndividual_N_28_M_28
      parameter Integer N = 28 "Number of segments of the primary distribution line";
      parameter Integer M = 28 "Number of segments of each secondary distribution line";
      parameter Real alpha = 10 "Distribution line oversizing factor";
      parameter Real beta = 2 "Ratio between line inductance and line resistance";
      parameter Modelica.SIunits.Resistance R_l = 100 "Resistance of a single load";
      parameter Modelica.SIunits.Resistance R_d2 = R_l / (M ^ 2 * alpha) "Resistance of a secondary distribution segment";
      parameter Modelica.SIunits.Resistance R_d1 = R_l / (M ^ 2 * N ^ 2 * alpha) "Resistance of a primary distribution segment";
      parameter Modelica.SIunits.Voltage V_ref = 600 "Reference source voltage";
      Models.Internals.Impedance primary_1(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_1_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_1_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_1_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_2(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_2_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_2_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_2_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_3(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_3_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_3_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_3_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_4(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_4_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_4_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_4_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_5(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_5_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_5_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_5_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_6(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_6_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_6_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_6_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_7(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_7_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_7_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_7_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_8(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_8_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_8_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_8_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_9(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_9_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_9_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_9_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_10(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_10_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_10_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_10_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_11(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_11_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_11_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_11_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_12(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_12_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_12_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_12_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_13(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_13_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_13_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_13_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_14(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_14_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_14_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_14_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_15(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_15_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_15_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_15_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_16(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_16_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_16_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_16_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_17(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_17_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_17_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_17_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_18(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_18_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_18_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_18_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_19(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_19_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_19_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_19_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_20(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_20_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_20_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_20_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_21(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_21_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_21_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_21_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_22(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_22_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_22_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_22_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_23(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_23_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_23_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_23_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_24(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_24_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_24_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_24_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_25(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_25_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_25_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_25_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_26(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_26_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_26_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_26_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_27(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_27_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_27_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_27_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance primary_28(Z(re = R_d1, im = R_d1 * beta)) "Primary distribution line segment";
      Models.Internals.Impedance secondary_28_1(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_1(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_2(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_2(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_3(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_3(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_4(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_4(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_5(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_5(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_6(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_6(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_7(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_7(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_8(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_8(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_9(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_9(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_10(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_10(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_11(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_11(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_12(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_12(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_13(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_13(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_14(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_14(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_15(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_15(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_16(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_16(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_17(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_17(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_18(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_18(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_19(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_19(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_20(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_20(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_21(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_21(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_22(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_22(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_23(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_23(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_24(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_24(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_25(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_25(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_26(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_26(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_27(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_27(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Impedance secondary_28_28(Z(re = R_d2, im = R_d2 * beta)) "Secondary distribution line segment";
      Models.Internals.LinearControlledLoad load_28_28(V_nom = V_ref, P_nom = V_ref ^ 2 / R_l) "Individual load resistor";
      Models.Internals.Ground sourceGround "Source ground";
      Models.Internals.VoltageSource V_source(V = V_ref) "Voltage source";
    equation
      connect(primary_1.p, V_source.p);
      connect(sourceGround.p, V_source.n);
      connect(primary_1.n, primary_2.p);
      connect(primary_2.n, primary_3.p);
      connect(primary_3.n, primary_4.p);
      connect(primary_4.n, primary_5.p);
      connect(primary_5.n, primary_6.p);
      connect(primary_6.n, primary_7.p);
      connect(primary_7.n, primary_8.p);
      connect(primary_8.n, primary_9.p);
      connect(primary_9.n, primary_10.p);
      connect(primary_10.n, primary_11.p);
      connect(primary_11.n, primary_12.p);
      connect(primary_12.n, primary_13.p);
      connect(primary_13.n, primary_14.p);
      connect(primary_14.n, primary_15.p);
      connect(primary_15.n, primary_16.p);
      connect(primary_16.n, primary_17.p);
      connect(primary_17.n, primary_18.p);
      connect(primary_18.n, primary_19.p);
      connect(primary_19.n, primary_20.p);
      connect(primary_20.n, primary_21.p);
      connect(primary_21.n, primary_22.p);
      connect(primary_22.n, primary_23.p);
      connect(primary_23.n, primary_24.p);
      connect(primary_24.n, primary_25.p);
      connect(primary_25.n, primary_26.p);
      connect(primary_26.n, primary_27.p);
      connect(primary_27.n, primary_28.p);
      connect(primary_1.n, secondary_1_1.p);
      connect(secondary_1_1.n, secondary_1_2.p);
      connect(secondary_1_2.n, secondary_1_3.p);
      connect(secondary_1_3.n, secondary_1_4.p);
      connect(secondary_1_4.n, secondary_1_5.p);
      connect(secondary_1_5.n, secondary_1_6.p);
      connect(secondary_1_6.n, secondary_1_7.p);
      connect(secondary_1_7.n, secondary_1_8.p);
      connect(secondary_1_8.n, secondary_1_9.p);
      connect(secondary_1_9.n, secondary_1_10.p);
      connect(secondary_1_10.n, secondary_1_11.p);
      connect(secondary_1_11.n, secondary_1_12.p);
      connect(secondary_1_12.n, secondary_1_13.p);
      connect(secondary_1_13.n, secondary_1_14.p);
      connect(secondary_1_14.n, secondary_1_15.p);
      connect(secondary_1_15.n, secondary_1_16.p);
      connect(secondary_1_16.n, secondary_1_17.p);
      connect(secondary_1_17.n, secondary_1_18.p);
      connect(secondary_1_18.n, secondary_1_19.p);
      connect(secondary_1_19.n, secondary_1_20.p);
      connect(secondary_1_20.n, secondary_1_21.p);
      connect(secondary_1_21.n, secondary_1_22.p);
      connect(secondary_1_22.n, secondary_1_23.p);
      connect(secondary_1_23.n, secondary_1_24.p);
      connect(secondary_1_24.n, secondary_1_25.p);
      connect(secondary_1_25.n, secondary_1_26.p);
      connect(secondary_1_26.n, secondary_1_27.p);
      connect(secondary_1_27.n, secondary_1_28.p);
      connect(secondary_1_1.n, load_1_1.p);
      connect(secondary_1_2.n, load_1_2.p);
      connect(secondary_1_3.n, load_1_3.p);
      connect(secondary_1_4.n, load_1_4.p);
      connect(secondary_1_5.n, load_1_5.p);
      connect(secondary_1_6.n, load_1_6.p);
      connect(secondary_1_7.n, load_1_7.p);
      connect(secondary_1_8.n, load_1_8.p);
      connect(secondary_1_9.n, load_1_9.p);
      connect(secondary_1_10.n, load_1_10.p);
      connect(secondary_1_11.n, load_1_11.p);
      connect(secondary_1_12.n, load_1_12.p);
      connect(secondary_1_13.n, load_1_13.p);
      connect(secondary_1_14.n, load_1_14.p);
      connect(secondary_1_15.n, load_1_15.p);
      connect(secondary_1_16.n, load_1_16.p);
      connect(secondary_1_17.n, load_1_17.p);
      connect(secondary_1_18.n, load_1_18.p);
      connect(secondary_1_19.n, load_1_19.p);
      connect(secondary_1_20.n, load_1_20.p);
      connect(secondary_1_21.n, load_1_21.p);
      connect(secondary_1_22.n, load_1_22.p);
      connect(secondary_1_23.n, load_1_23.p);
      connect(secondary_1_24.n, load_1_24.p);
      connect(secondary_1_25.n, load_1_25.p);
      connect(secondary_1_26.n, load_1_26.p);
      connect(secondary_1_27.n, load_1_27.p);
      connect(secondary_1_28.n, load_1_28.p);
      connect(primary_2.n, secondary_2_1.p);
      connect(secondary_2_1.n, secondary_2_2.p);
      connect(secondary_2_2.n, secondary_2_3.p);
      connect(secondary_2_3.n, secondary_2_4.p);
      connect(secondary_2_4.n, secondary_2_5.p);
      connect(secondary_2_5.n, secondary_2_6.p);
      connect(secondary_2_6.n, secondary_2_7.p);
      connect(secondary_2_7.n, secondary_2_8.p);
      connect(secondary_2_8.n, secondary_2_9.p);
      connect(secondary_2_9.n, secondary_2_10.p);
      connect(secondary_2_10.n, secondary_2_11.p);
      connect(secondary_2_11.n, secondary_2_12.p);
      connect(secondary_2_12.n, secondary_2_13.p);
      connect(secondary_2_13.n, secondary_2_14.p);
      connect(secondary_2_14.n, secondary_2_15.p);
      connect(secondary_2_15.n, secondary_2_16.p);
      connect(secondary_2_16.n, secondary_2_17.p);
      connect(secondary_2_17.n, secondary_2_18.p);
      connect(secondary_2_18.n, secondary_2_19.p);
      connect(secondary_2_19.n, secondary_2_20.p);
      connect(secondary_2_20.n, secondary_2_21.p);
      connect(secondary_2_21.n, secondary_2_22.p);
      connect(secondary_2_22.n, secondary_2_23.p);
      connect(secondary_2_23.n, secondary_2_24.p);
      connect(secondary_2_24.n, secondary_2_25.p);
      connect(secondary_2_25.n, secondary_2_26.p);
      connect(secondary_2_26.n, secondary_2_27.p);
      connect(secondary_2_27.n, secondary_2_28.p);
      connect(secondary_2_1.n, load_2_1.p);
      connect(secondary_2_2.n, load_2_2.p);
      connect(secondary_2_3.n, load_2_3.p);
      connect(secondary_2_4.n, load_2_4.p);
      connect(secondary_2_5.n, load_2_5.p);
      connect(secondary_2_6.n, load_2_6.p);
      connect(secondary_2_7.n, load_2_7.p);
      connect(secondary_2_8.n, load_2_8.p);
      connect(secondary_2_9.n, load_2_9.p);
      connect(secondary_2_10.n, load_2_10.p);
      connect(secondary_2_11.n, load_2_11.p);
      connect(secondary_2_12.n, load_2_12.p);
      connect(secondary_2_13.n, load_2_13.p);
      connect(secondary_2_14.n, load_2_14.p);
      connect(secondary_2_15.n, load_2_15.p);
      connect(secondary_2_16.n, load_2_16.p);
      connect(secondary_2_17.n, load_2_17.p);
      connect(secondary_2_18.n, load_2_18.p);
      connect(secondary_2_19.n, load_2_19.p);
      connect(secondary_2_20.n, load_2_20.p);
      connect(secondary_2_21.n, load_2_21.p);
      connect(secondary_2_22.n, load_2_22.p);
      connect(secondary_2_23.n, load_2_23.p);
      connect(secondary_2_24.n, load_2_24.p);
      connect(secondary_2_25.n, load_2_25.p);
      connect(secondary_2_26.n, load_2_26.p);
      connect(secondary_2_27.n, load_2_27.p);
      connect(secondary_2_28.n, load_2_28.p);
      connect(primary_3.n, secondary_3_1.p);
      connect(secondary_3_1.n, secondary_3_2.p);
      connect(secondary_3_2.n, secondary_3_3.p);
      connect(secondary_3_3.n, secondary_3_4.p);
      connect(secondary_3_4.n, secondary_3_5.p);
      connect(secondary_3_5.n, secondary_3_6.p);
      connect(secondary_3_6.n, secondary_3_7.p);
      connect(secondary_3_7.n, secondary_3_8.p);
      connect(secondary_3_8.n, secondary_3_9.p);
      connect(secondary_3_9.n, secondary_3_10.p);
      connect(secondary_3_10.n, secondary_3_11.p);
      connect(secondary_3_11.n, secondary_3_12.p);
      connect(secondary_3_12.n, secondary_3_13.p);
      connect(secondary_3_13.n, secondary_3_14.p);
      connect(secondary_3_14.n, secondary_3_15.p);
      connect(secondary_3_15.n, secondary_3_16.p);
      connect(secondary_3_16.n, secondary_3_17.p);
      connect(secondary_3_17.n, secondary_3_18.p);
      connect(secondary_3_18.n, secondary_3_19.p);
      connect(secondary_3_19.n, secondary_3_20.p);
      connect(secondary_3_20.n, secondary_3_21.p);
      connect(secondary_3_21.n, secondary_3_22.p);
      connect(secondary_3_22.n, secondary_3_23.p);
      connect(secondary_3_23.n, secondary_3_24.p);
      connect(secondary_3_24.n, secondary_3_25.p);
      connect(secondary_3_25.n, secondary_3_26.p);
      connect(secondary_3_26.n, secondary_3_27.p);
      connect(secondary_3_27.n, secondary_3_28.p);
      connect(secondary_3_1.n, load_3_1.p);
      connect(secondary_3_2.n, load_3_2.p);
      connect(secondary_3_3.n, load_3_3.p);
      connect(secondary_3_4.n, load_3_4.p);
      connect(secondary_3_5.n, load_3_5.p);
      connect(secondary_3_6.n, load_3_6.p);
      connect(secondary_3_7.n, load_3_7.p);
      connect(secondary_3_8.n, load_3_8.p);
      connect(secondary_3_9.n, load_3_9.p);
      connect(secondary_3_10.n, load_3_10.p);
      connect(secondary_3_11.n, load_3_11.p);
      connect(secondary_3_12.n, load_3_12.p);
      connect(secondary_3_13.n, load_3_13.p);
      connect(secondary_3_14.n, load_3_14.p);
      connect(secondary_3_15.n, load_3_15.p);
      connect(secondary_3_16.n, load_3_16.p);
      connect(secondary_3_17.n, load_3_17.p);
      connect(secondary_3_18.n, load_3_18.p);
      connect(secondary_3_19.n, load_3_19.p);
      connect(secondary_3_20.n, load_3_20.p);
      connect(secondary_3_21.n, load_3_21.p);
      connect(secondary_3_22.n, load_3_22.p);
      connect(secondary_3_23.n, load_3_23.p);
      connect(secondary_3_24.n, load_3_24.p);
      connect(secondary_3_25.n, load_3_25.p);
      connect(secondary_3_26.n, load_3_26.p);
      connect(secondary_3_27.n, load_3_27.p);
      connect(secondary_3_28.n, load_3_28.p);
      connect(primary_4.n, secondary_4_1.p);
      connect(secondary_4_1.n, secondary_4_2.p);
      connect(secondary_4_2.n, secondary_4_3.p);
      connect(secondary_4_3.n, secondary_4_4.p);
      connect(secondary_4_4.n, secondary_4_5.p);
      connect(secondary_4_5.n, secondary_4_6.p);
      connect(secondary_4_6.n, secondary_4_7.p);
      connect(secondary_4_7.n, secondary_4_8.p);
      connect(secondary_4_8.n, secondary_4_9.p);
      connect(secondary_4_9.n, secondary_4_10.p);
      connect(secondary_4_10.n, secondary_4_11.p);
      connect(secondary_4_11.n, secondary_4_12.p);
      connect(secondary_4_12.n, secondary_4_13.p);
      connect(secondary_4_13.n, secondary_4_14.p);
      connect(secondary_4_14.n, secondary_4_15.p);
      connect(secondary_4_15.n, secondary_4_16.p);
      connect(secondary_4_16.n, secondary_4_17.p);
      connect(secondary_4_17.n, secondary_4_18.p);
      connect(secondary_4_18.n, secondary_4_19.p);
      connect(secondary_4_19.n, secondary_4_20.p);
      connect(secondary_4_20.n, secondary_4_21.p);
      connect(secondary_4_21.n, secondary_4_22.p);
      connect(secondary_4_22.n, secondary_4_23.p);
      connect(secondary_4_23.n, secondary_4_24.p);
      connect(secondary_4_24.n, secondary_4_25.p);
      connect(secondary_4_25.n, secondary_4_26.p);
      connect(secondary_4_26.n, secondary_4_27.p);
      connect(secondary_4_27.n, secondary_4_28.p);
      connect(secondary_4_1.n, load_4_1.p);
      connect(secondary_4_2.n, load_4_2.p);
      connect(secondary_4_3.n, load_4_3.p);
      connect(secondary_4_4.n, load_4_4.p);
      connect(secondary_4_5.n, load_4_5.p);
      connect(secondary_4_6.n, load_4_6.p);
      connect(secondary_4_7.n, load_4_7.p);
      connect(secondary_4_8.n, load_4_8.p);
      connect(secondary_4_9.n, load_4_9.p);
      connect(secondary_4_10.n, load_4_10.p);
      connect(secondary_4_11.n, load_4_11.p);
      connect(secondary_4_12.n, load_4_12.p);
      connect(secondary_4_13.n, load_4_13.p);
      connect(secondary_4_14.n, load_4_14.p);
      connect(secondary_4_15.n, load_4_15.p);
      connect(secondary_4_16.n, load_4_16.p);
      connect(secondary_4_17.n, load_4_17.p);
      connect(secondary_4_18.n, load_4_18.p);
      connect(secondary_4_19.n, load_4_19.p);
      connect(secondary_4_20.n, load_4_20.p);
      connect(secondary_4_21.n, load_4_21.p);
      connect(secondary_4_22.n, load_4_22.p);
      connect(secondary_4_23.n, load_4_23.p);
      connect(secondary_4_24.n, load_4_24.p);
      connect(secondary_4_25.n, load_4_25.p);
      connect(secondary_4_26.n, load_4_26.p);
      connect(secondary_4_27.n, load_4_27.p);
      connect(secondary_4_28.n, load_4_28.p);
      connect(primary_5.n, secondary_5_1.p);
      connect(secondary_5_1.n, secondary_5_2.p);
      connect(secondary_5_2.n, secondary_5_3.p);
      connect(secondary_5_3.n, secondary_5_4.p);
      connect(secondary_5_4.n, secondary_5_5.p);
      connect(secondary_5_5.n, secondary_5_6.p);
      connect(secondary_5_6.n, secondary_5_7.p);
      connect(secondary_5_7.n, secondary_5_8.p);
      connect(secondary_5_8.n, secondary_5_9.p);
      connect(secondary_5_9.n, secondary_5_10.p);
      connect(secondary_5_10.n, secondary_5_11.p);
      connect(secondary_5_11.n, secondary_5_12.p);
      connect(secondary_5_12.n, secondary_5_13.p);
      connect(secondary_5_13.n, secondary_5_14.p);
      connect(secondary_5_14.n, secondary_5_15.p);
      connect(secondary_5_15.n, secondary_5_16.p);
      connect(secondary_5_16.n, secondary_5_17.p);
      connect(secondary_5_17.n, secondary_5_18.p);
      connect(secondary_5_18.n, secondary_5_19.p);
      connect(secondary_5_19.n, secondary_5_20.p);
      connect(secondary_5_20.n, secondary_5_21.p);
      connect(secondary_5_21.n, secondary_5_22.p);
      connect(secondary_5_22.n, secondary_5_23.p);
      connect(secondary_5_23.n, secondary_5_24.p);
      connect(secondary_5_24.n, secondary_5_25.p);
      connect(secondary_5_25.n, secondary_5_26.p);
      connect(secondary_5_26.n, secondary_5_27.p);
      connect(secondary_5_27.n, secondary_5_28.p);
      connect(secondary_5_1.n, load_5_1.p);
      connect(secondary_5_2.n, load_5_2.p);
      connect(secondary_5_3.n, load_5_3.p);
      connect(secondary_5_4.n, load_5_4.p);
      connect(secondary_5_5.n, load_5_5.p);
      connect(secondary_5_6.n, load_5_6.p);
      connect(secondary_5_7.n, load_5_7.p);
      connect(secondary_5_8.n, load_5_8.p);
      connect(secondary_5_9.n, load_5_9.p);
      connect(secondary_5_10.n, load_5_10.p);
      connect(secondary_5_11.n, load_5_11.p);
      connect(secondary_5_12.n, load_5_12.p);
      connect(secondary_5_13.n, load_5_13.p);
      connect(secondary_5_14.n, load_5_14.p);
      connect(secondary_5_15.n, load_5_15.p);
      connect(secondary_5_16.n, load_5_16.p);
      connect(secondary_5_17.n, load_5_17.p);
      connect(secondary_5_18.n, load_5_18.p);
      connect(secondary_5_19.n, load_5_19.p);
      connect(secondary_5_20.n, load_5_20.p);
      connect(secondary_5_21.n, load_5_21.p);
      connect(secondary_5_22.n, load_5_22.p);
      connect(secondary_5_23.n, load_5_23.p);
      connect(secondary_5_24.n, load_5_24.p);
      connect(secondary_5_25.n, load_5_25.p);
      connect(secondary_5_26.n, load_5_26.p);
      connect(secondary_5_27.n, load_5_27.p);
      connect(secondary_5_28.n, load_5_28.p);
      connect(primary_6.n, secondary_6_1.p);
      connect(secondary_6_1.n, secondary_6_2.p);
      connect(secondary_6_2.n, secondary_6_3.p);
      connect(secondary_6_3.n, secondary_6_4.p);
      connect(secondary_6_4.n, secondary_6_5.p);
      connect(secondary_6_5.n, secondary_6_6.p);
      connect(secondary_6_6.n, secondary_6_7.p);
      connect(secondary_6_7.n, secondary_6_8.p);
      connect(secondary_6_8.n, secondary_6_9.p);
      connect(secondary_6_9.n, secondary_6_10.p);
      connect(secondary_6_10.n, secondary_6_11.p);
      connect(secondary_6_11.n, secondary_6_12.p);
      connect(secondary_6_12.n, secondary_6_13.p);
      connect(secondary_6_13.n, secondary_6_14.p);
      connect(secondary_6_14.n, secondary_6_15.p);
      connect(secondary_6_15.n, secondary_6_16.p);
      connect(secondary_6_16.n, secondary_6_17.p);
      connect(secondary_6_17.n, secondary_6_18.p);
      connect(secondary_6_18.n, secondary_6_19.p);
      connect(secondary_6_19.n, secondary_6_20.p);
      connect(secondary_6_20.n, secondary_6_21.p);
      connect(secondary_6_21.n, secondary_6_22.p);
      connect(secondary_6_22.n, secondary_6_23.p);
      connect(secondary_6_23.n, secondary_6_24.p);
      connect(secondary_6_24.n, secondary_6_25.p);
      connect(secondary_6_25.n, secondary_6_26.p);
      connect(secondary_6_26.n, secondary_6_27.p);
      connect(secondary_6_27.n, secondary_6_28.p);
      connect(secondary_6_1.n, load_6_1.p);
      connect(secondary_6_2.n, load_6_2.p);
      connect(secondary_6_3.n, load_6_3.p);
      connect(secondary_6_4.n, load_6_4.p);
      connect(secondary_6_5.n, load_6_5.p);
      connect(secondary_6_6.n, load_6_6.p);
      connect(secondary_6_7.n, load_6_7.p);
      connect(secondary_6_8.n, load_6_8.p);
      connect(secondary_6_9.n, load_6_9.p);
      connect(secondary_6_10.n, load_6_10.p);
      connect(secondary_6_11.n, load_6_11.p);
      connect(secondary_6_12.n, load_6_12.p);
      connect(secondary_6_13.n, load_6_13.p);
      connect(secondary_6_14.n, load_6_14.p);
      connect(secondary_6_15.n, load_6_15.p);
      connect(secondary_6_16.n, load_6_16.p);
      connect(secondary_6_17.n, load_6_17.p);
      connect(secondary_6_18.n, load_6_18.p);
      connect(secondary_6_19.n, load_6_19.p);
      connect(secondary_6_20.n, load_6_20.p);
      connect(secondary_6_21.n, load_6_21.p);
      connect(secondary_6_22.n, load_6_22.p);
      connect(secondary_6_23.n, load_6_23.p);
      connect(secondary_6_24.n, load_6_24.p);
      connect(secondary_6_25.n, load_6_25.p);
      connect(secondary_6_26.n, load_6_26.p);
      connect(secondary_6_27.n, load_6_27.p);
      connect(secondary_6_28.n, load_6_28.p);
      connect(primary_7.n, secondary_7_1.p);
      connect(secondary_7_1.n, secondary_7_2.p);
      connect(secondary_7_2.n, secondary_7_3.p);
      connect(secondary_7_3.n, secondary_7_4.p);
      connect(secondary_7_4.n, secondary_7_5.p);
      connect(secondary_7_5.n, secondary_7_6.p);
      connect(secondary_7_6.n, secondary_7_7.p);
      connect(secondary_7_7.n, secondary_7_8.p);
      connect(secondary_7_8.n, secondary_7_9.p);
      connect(secondary_7_9.n, secondary_7_10.p);
      connect(secondary_7_10.n, secondary_7_11.p);
      connect(secondary_7_11.n, secondary_7_12.p);
      connect(secondary_7_12.n, secondary_7_13.p);
      connect(secondary_7_13.n, secondary_7_14.p);
      connect(secondary_7_14.n, secondary_7_15.p);
      connect(secondary_7_15.n, secondary_7_16.p);
      connect(secondary_7_16.n, secondary_7_17.p);
      connect(secondary_7_17.n, secondary_7_18.p);
      connect(secondary_7_18.n, secondary_7_19.p);
      connect(secondary_7_19.n, secondary_7_20.p);
      connect(secondary_7_20.n, secondary_7_21.p);
      connect(secondary_7_21.n, secondary_7_22.p);
      connect(secondary_7_22.n, secondary_7_23.p);
      connect(secondary_7_23.n, secondary_7_24.p);
      connect(secondary_7_24.n, secondary_7_25.p);
      connect(secondary_7_25.n, secondary_7_26.p);
      connect(secondary_7_26.n, secondary_7_27.p);
      connect(secondary_7_27.n, secondary_7_28.p);
      connect(secondary_7_1.n, load_7_1.p);
      connect(secondary_7_2.n, load_7_2.p);
      connect(secondary_7_3.n, load_7_3.p);
      connect(secondary_7_4.n, load_7_4.p);
      connect(secondary_7_5.n, load_7_5.p);
      connect(secondary_7_6.n, load_7_6.p);
      connect(secondary_7_7.n, load_7_7.p);
      connect(secondary_7_8.n, load_7_8.p);
      connect(secondary_7_9.n, load_7_9.p);
      connect(secondary_7_10.n, load_7_10.p);
      connect(secondary_7_11.n, load_7_11.p);
      connect(secondary_7_12.n, load_7_12.p);
      connect(secondary_7_13.n, load_7_13.p);
      connect(secondary_7_14.n, load_7_14.p);
      connect(secondary_7_15.n, load_7_15.p);
      connect(secondary_7_16.n, load_7_16.p);
      connect(secondary_7_17.n, load_7_17.p);
      connect(secondary_7_18.n, load_7_18.p);
      connect(secondary_7_19.n, load_7_19.p);
      connect(secondary_7_20.n, load_7_20.p);
      connect(secondary_7_21.n, load_7_21.p);
      connect(secondary_7_22.n, load_7_22.p);
      connect(secondary_7_23.n, load_7_23.p);
      connect(secondary_7_24.n, load_7_24.p);
      connect(secondary_7_25.n, load_7_25.p);
      connect(secondary_7_26.n, load_7_26.p);
      connect(secondary_7_27.n, load_7_27.p);
      connect(secondary_7_28.n, load_7_28.p);
      connect(primary_8.n, secondary_8_1.p);
      connect(secondary_8_1.n, secondary_8_2.p);
      connect(secondary_8_2.n, secondary_8_3.p);
      connect(secondary_8_3.n, secondary_8_4.p);
      connect(secondary_8_4.n, secondary_8_5.p);
      connect(secondary_8_5.n, secondary_8_6.p);
      connect(secondary_8_6.n, secondary_8_7.p);
      connect(secondary_8_7.n, secondary_8_8.p);
      connect(secondary_8_8.n, secondary_8_9.p);
      connect(secondary_8_9.n, secondary_8_10.p);
      connect(secondary_8_10.n, secondary_8_11.p);
      connect(secondary_8_11.n, secondary_8_12.p);
      connect(secondary_8_12.n, secondary_8_13.p);
      connect(secondary_8_13.n, secondary_8_14.p);
      connect(secondary_8_14.n, secondary_8_15.p);
      connect(secondary_8_15.n, secondary_8_16.p);
      connect(secondary_8_16.n, secondary_8_17.p);
      connect(secondary_8_17.n, secondary_8_18.p);
      connect(secondary_8_18.n, secondary_8_19.p);
      connect(secondary_8_19.n, secondary_8_20.p);
      connect(secondary_8_20.n, secondary_8_21.p);
      connect(secondary_8_21.n, secondary_8_22.p);
      connect(secondary_8_22.n, secondary_8_23.p);
      connect(secondary_8_23.n, secondary_8_24.p);
      connect(secondary_8_24.n, secondary_8_25.p);
      connect(secondary_8_25.n, secondary_8_26.p);
      connect(secondary_8_26.n, secondary_8_27.p);
      connect(secondary_8_27.n, secondary_8_28.p);
      connect(secondary_8_1.n, load_8_1.p);
      connect(secondary_8_2.n, load_8_2.p);
      connect(secondary_8_3.n, load_8_3.p);
      connect(secondary_8_4.n, load_8_4.p);
      connect(secondary_8_5.n, load_8_5.p);
      connect(secondary_8_6.n, load_8_6.p);
      connect(secondary_8_7.n, load_8_7.p);
      connect(secondary_8_8.n, load_8_8.p);
      connect(secondary_8_9.n, load_8_9.p);
      connect(secondary_8_10.n, load_8_10.p);
      connect(secondary_8_11.n, load_8_11.p);
      connect(secondary_8_12.n, load_8_12.p);
      connect(secondary_8_13.n, load_8_13.p);
      connect(secondary_8_14.n, load_8_14.p);
      connect(secondary_8_15.n, load_8_15.p);
      connect(secondary_8_16.n, load_8_16.p);
      connect(secondary_8_17.n, load_8_17.p);
      connect(secondary_8_18.n, load_8_18.p);
      connect(secondary_8_19.n, load_8_19.p);
      connect(secondary_8_20.n, load_8_20.p);
      connect(secondary_8_21.n, load_8_21.p);
      connect(secondary_8_22.n, load_8_22.p);
      connect(secondary_8_23.n, load_8_23.p);
      connect(secondary_8_24.n, load_8_24.p);
      connect(secondary_8_25.n, load_8_25.p);
      connect(secondary_8_26.n, load_8_26.p);
      connect(secondary_8_27.n, load_8_27.p);
      connect(secondary_8_28.n, load_8_28.p);
      connect(primary_9.n, secondary_9_1.p);
      connect(secondary_9_1.n, secondary_9_2.p);
      connect(secondary_9_2.n, secondary_9_3.p);
      connect(secondary_9_3.n, secondary_9_4.p);
      connect(secondary_9_4.n, secondary_9_5.p);
      connect(secondary_9_5.n, secondary_9_6.p);
      connect(secondary_9_6.n, secondary_9_7.p);
      connect(secondary_9_7.n, secondary_9_8.p);
      connect(secondary_9_8.n, secondary_9_9.p);
      connect(secondary_9_9.n, secondary_9_10.p);
      connect(secondary_9_10.n, secondary_9_11.p);
      connect(secondary_9_11.n, secondary_9_12.p);
      connect(secondary_9_12.n, secondary_9_13.p);
      connect(secondary_9_13.n, secondary_9_14.p);
      connect(secondary_9_14.n, secondary_9_15.p);
      connect(secondary_9_15.n, secondary_9_16.p);
      connect(secondary_9_16.n, secondary_9_17.p);
      connect(secondary_9_17.n, secondary_9_18.p);
      connect(secondary_9_18.n, secondary_9_19.p);
      connect(secondary_9_19.n, secondary_9_20.p);
      connect(secondary_9_20.n, secondary_9_21.p);
      connect(secondary_9_21.n, secondary_9_22.p);
      connect(secondary_9_22.n, secondary_9_23.p);
      connect(secondary_9_23.n, secondary_9_24.p);
      connect(secondary_9_24.n, secondary_9_25.p);
      connect(secondary_9_25.n, secondary_9_26.p);
      connect(secondary_9_26.n, secondary_9_27.p);
      connect(secondary_9_27.n, secondary_9_28.p);
      connect(secondary_9_1.n, load_9_1.p);
      connect(secondary_9_2.n, load_9_2.p);
      connect(secondary_9_3.n, load_9_3.p);
      connect(secondary_9_4.n, load_9_4.p);
      connect(secondary_9_5.n, load_9_5.p);
      connect(secondary_9_6.n, load_9_6.p);
      connect(secondary_9_7.n, load_9_7.p);
      connect(secondary_9_8.n, load_9_8.p);
      connect(secondary_9_9.n, load_9_9.p);
      connect(secondary_9_10.n, load_9_10.p);
      connect(secondary_9_11.n, load_9_11.p);
      connect(secondary_9_12.n, load_9_12.p);
      connect(secondary_9_13.n, load_9_13.p);
      connect(secondary_9_14.n, load_9_14.p);
      connect(secondary_9_15.n, load_9_15.p);
      connect(secondary_9_16.n, load_9_16.p);
      connect(secondary_9_17.n, load_9_17.p);
      connect(secondary_9_18.n, load_9_18.p);
      connect(secondary_9_19.n, load_9_19.p);
      connect(secondary_9_20.n, load_9_20.p);
      connect(secondary_9_21.n, load_9_21.p);
      connect(secondary_9_22.n, load_9_22.p);
      connect(secondary_9_23.n, load_9_23.p);
      connect(secondary_9_24.n, load_9_24.p);
      connect(secondary_9_25.n, load_9_25.p);
      connect(secondary_9_26.n, load_9_26.p);
      connect(secondary_9_27.n, load_9_27.p);
      connect(secondary_9_28.n, load_9_28.p);
      connect(primary_10.n, secondary_10_1.p);
      connect(secondary_10_1.n, secondary_10_2.p);
      connect(secondary_10_2.n, secondary_10_3.p);
      connect(secondary_10_3.n, secondary_10_4.p);
      connect(secondary_10_4.n, secondary_10_5.p);
      connect(secondary_10_5.n, secondary_10_6.p);
      connect(secondary_10_6.n, secondary_10_7.p);
      connect(secondary_10_7.n, secondary_10_8.p);
      connect(secondary_10_8.n, secondary_10_9.p);
      connect(secondary_10_9.n, secondary_10_10.p);
      connect(secondary_10_10.n, secondary_10_11.p);
      connect(secondary_10_11.n, secondary_10_12.p);
      connect(secondary_10_12.n, secondary_10_13.p);
      connect(secondary_10_13.n, secondary_10_14.p);
      connect(secondary_10_14.n, secondary_10_15.p);
      connect(secondary_10_15.n, secondary_10_16.p);
      connect(secondary_10_16.n, secondary_10_17.p);
      connect(secondary_10_17.n, secondary_10_18.p);
      connect(secondary_10_18.n, secondary_10_19.p);
      connect(secondary_10_19.n, secondary_10_20.p);
      connect(secondary_10_20.n, secondary_10_21.p);
      connect(secondary_10_21.n, secondary_10_22.p);
      connect(secondary_10_22.n, secondary_10_23.p);
      connect(secondary_10_23.n, secondary_10_24.p);
      connect(secondary_10_24.n, secondary_10_25.p);
      connect(secondary_10_25.n, secondary_10_26.p);
      connect(secondary_10_26.n, secondary_10_27.p);
      connect(secondary_10_27.n, secondary_10_28.p);
      connect(secondary_10_1.n, load_10_1.p);
      connect(secondary_10_2.n, load_10_2.p);
      connect(secondary_10_3.n, load_10_3.p);
      connect(secondary_10_4.n, load_10_4.p);
      connect(secondary_10_5.n, load_10_5.p);
      connect(secondary_10_6.n, load_10_6.p);
      connect(secondary_10_7.n, load_10_7.p);
      connect(secondary_10_8.n, load_10_8.p);
      connect(secondary_10_9.n, load_10_9.p);
      connect(secondary_10_10.n, load_10_10.p);
      connect(secondary_10_11.n, load_10_11.p);
      connect(secondary_10_12.n, load_10_12.p);
      connect(secondary_10_13.n, load_10_13.p);
      connect(secondary_10_14.n, load_10_14.p);
      connect(secondary_10_15.n, load_10_15.p);
      connect(secondary_10_16.n, load_10_16.p);
      connect(secondary_10_17.n, load_10_17.p);
      connect(secondary_10_18.n, load_10_18.p);
      connect(secondary_10_19.n, load_10_19.p);
      connect(secondary_10_20.n, load_10_20.p);
      connect(secondary_10_21.n, load_10_21.p);
      connect(secondary_10_22.n, load_10_22.p);
      connect(secondary_10_23.n, load_10_23.p);
      connect(secondary_10_24.n, load_10_24.p);
      connect(secondary_10_25.n, load_10_25.p);
      connect(secondary_10_26.n, load_10_26.p);
      connect(secondary_10_27.n, load_10_27.p);
      connect(secondary_10_28.n, load_10_28.p);
      connect(primary_11.n, secondary_11_1.p);
      connect(secondary_11_1.n, secondary_11_2.p);
      connect(secondary_11_2.n, secondary_11_3.p);
      connect(secondary_11_3.n, secondary_11_4.p);
      connect(secondary_11_4.n, secondary_11_5.p);
      connect(secondary_11_5.n, secondary_11_6.p);
      connect(secondary_11_6.n, secondary_11_7.p);
      connect(secondary_11_7.n, secondary_11_8.p);
      connect(secondary_11_8.n, secondary_11_9.p);
      connect(secondary_11_9.n, secondary_11_10.p);
      connect(secondary_11_10.n, secondary_11_11.p);
      connect(secondary_11_11.n, secondary_11_12.p);
      connect(secondary_11_12.n, secondary_11_13.p);
      connect(secondary_11_13.n, secondary_11_14.p);
      connect(secondary_11_14.n, secondary_11_15.p);
      connect(secondary_11_15.n, secondary_11_16.p);
      connect(secondary_11_16.n, secondary_11_17.p);
      connect(secondary_11_17.n, secondary_11_18.p);
      connect(secondary_11_18.n, secondary_11_19.p);
      connect(secondary_11_19.n, secondary_11_20.p);
      connect(secondary_11_20.n, secondary_11_21.p);
      connect(secondary_11_21.n, secondary_11_22.p);
      connect(secondary_11_22.n, secondary_11_23.p);
      connect(secondary_11_23.n, secondary_11_24.p);
      connect(secondary_11_24.n, secondary_11_25.p);
      connect(secondary_11_25.n, secondary_11_26.p);
      connect(secondary_11_26.n, secondary_11_27.p);
      connect(secondary_11_27.n, secondary_11_28.p);
      connect(secondary_11_1.n, load_11_1.p);
      connect(secondary_11_2.n, load_11_2.p);
      connect(secondary_11_3.n, load_11_3.p);
      connect(secondary_11_4.n, load_11_4.p);
      connect(secondary_11_5.n, load_11_5.p);
      connect(secondary_11_6.n, load_11_6.p);
      connect(secondary_11_7.n, load_11_7.p);
      connect(secondary_11_8.n, load_11_8.p);
      connect(secondary_11_9.n, load_11_9.p);
      connect(secondary_11_10.n, load_11_10.p);
      connect(secondary_11_11.n, load_11_11.p);
      connect(secondary_11_12.n, load_11_12.p);
      connect(secondary_11_13.n, load_11_13.p);
      connect(secondary_11_14.n, load_11_14.p);
      connect(secondary_11_15.n, load_11_15.p);
      connect(secondary_11_16.n, load_11_16.p);
      connect(secondary_11_17.n, load_11_17.p);
      connect(secondary_11_18.n, load_11_18.p);
      connect(secondary_11_19.n, load_11_19.p);
      connect(secondary_11_20.n, load_11_20.p);
      connect(secondary_11_21.n, load_11_21.p);
      connect(secondary_11_22.n, load_11_22.p);
      connect(secondary_11_23.n, load_11_23.p);
      connect(secondary_11_24.n, load_11_24.p);
      connect(secondary_11_25.n, load_11_25.p);
      connect(secondary_11_26.n, load_11_26.p);
      connect(secondary_11_27.n, load_11_27.p);
      connect(secondary_11_28.n, load_11_28.p);
      connect(primary_12.n, secondary_12_1.p);
      connect(secondary_12_1.n, secondary_12_2.p);
      connect(secondary_12_2.n, secondary_12_3.p);
      connect(secondary_12_3.n, secondary_12_4.p);
      connect(secondary_12_4.n, secondary_12_5.p);
      connect(secondary_12_5.n, secondary_12_6.p);
      connect(secondary_12_6.n, secondary_12_7.p);
      connect(secondary_12_7.n, secondary_12_8.p);
      connect(secondary_12_8.n, secondary_12_9.p);
      connect(secondary_12_9.n, secondary_12_10.p);
      connect(secondary_12_10.n, secondary_12_11.p);
      connect(secondary_12_11.n, secondary_12_12.p);
      connect(secondary_12_12.n, secondary_12_13.p);
      connect(secondary_12_13.n, secondary_12_14.p);
      connect(secondary_12_14.n, secondary_12_15.p);
      connect(secondary_12_15.n, secondary_12_16.p);
      connect(secondary_12_16.n, secondary_12_17.p);
      connect(secondary_12_17.n, secondary_12_18.p);
      connect(secondary_12_18.n, secondary_12_19.p);
      connect(secondary_12_19.n, secondary_12_20.p);
      connect(secondary_12_20.n, secondary_12_21.p);
      connect(secondary_12_21.n, secondary_12_22.p);
      connect(secondary_12_22.n, secondary_12_23.p);
      connect(secondary_12_23.n, secondary_12_24.p);
      connect(secondary_12_24.n, secondary_12_25.p);
      connect(secondary_12_25.n, secondary_12_26.p);
      connect(secondary_12_26.n, secondary_12_27.p);
      connect(secondary_12_27.n, secondary_12_28.p);
      connect(secondary_12_1.n, load_12_1.p);
      connect(secondary_12_2.n, load_12_2.p);
      connect(secondary_12_3.n, load_12_3.p);
      connect(secondary_12_4.n, load_12_4.p);
      connect(secondary_12_5.n, load_12_5.p);
      connect(secondary_12_6.n, load_12_6.p);
      connect(secondary_12_7.n, load_12_7.p);
      connect(secondary_12_8.n, load_12_8.p);
      connect(secondary_12_9.n, load_12_9.p);
      connect(secondary_12_10.n, load_12_10.p);
      connect(secondary_12_11.n, load_12_11.p);
      connect(secondary_12_12.n, load_12_12.p);
      connect(secondary_12_13.n, load_12_13.p);
      connect(secondary_12_14.n, load_12_14.p);
      connect(secondary_12_15.n, load_12_15.p);
      connect(secondary_12_16.n, load_12_16.p);
      connect(secondary_12_17.n, load_12_17.p);
      connect(secondary_12_18.n, load_12_18.p);
      connect(secondary_12_19.n, load_12_19.p);
      connect(secondary_12_20.n, load_12_20.p);
      connect(secondary_12_21.n, load_12_21.p);
      connect(secondary_12_22.n, load_12_22.p);
      connect(secondary_12_23.n, load_12_23.p);
      connect(secondary_12_24.n, load_12_24.p);
      connect(secondary_12_25.n, load_12_25.p);
      connect(secondary_12_26.n, load_12_26.p);
      connect(secondary_12_27.n, load_12_27.p);
      connect(secondary_12_28.n, load_12_28.p);
      connect(primary_13.n, secondary_13_1.p);
      connect(secondary_13_1.n, secondary_13_2.p);
      connect(secondary_13_2.n, secondary_13_3.p);
      connect(secondary_13_3.n, secondary_13_4.p);
      connect(secondary_13_4.n, secondary_13_5.p);
      connect(secondary_13_5.n, secondary_13_6.p);
      connect(secondary_13_6.n, secondary_13_7.p);
      connect(secondary_13_7.n, secondary_13_8.p);
      connect(secondary_13_8.n, secondary_13_9.p);
      connect(secondary_13_9.n, secondary_13_10.p);
      connect(secondary_13_10.n, secondary_13_11.p);
      connect(secondary_13_11.n, secondary_13_12.p);
      connect(secondary_13_12.n, secondary_13_13.p);
      connect(secondary_13_13.n, secondary_13_14.p);
      connect(secondary_13_14.n, secondary_13_15.p);
      connect(secondary_13_15.n, secondary_13_16.p);
      connect(secondary_13_16.n, secondary_13_17.p);
      connect(secondary_13_17.n, secondary_13_18.p);
      connect(secondary_13_18.n, secondary_13_19.p);
      connect(secondary_13_19.n, secondary_13_20.p);
      connect(secondary_13_20.n, secondary_13_21.p);
      connect(secondary_13_21.n, secondary_13_22.p);
      connect(secondary_13_22.n, secondary_13_23.p);
      connect(secondary_13_23.n, secondary_13_24.p);
      connect(secondary_13_24.n, secondary_13_25.p);
      connect(secondary_13_25.n, secondary_13_26.p);
      connect(secondary_13_26.n, secondary_13_27.p);
      connect(secondary_13_27.n, secondary_13_28.p);
      connect(secondary_13_1.n, load_13_1.p);
      connect(secondary_13_2.n, load_13_2.p);
      connect(secondary_13_3.n, load_13_3.p);
      connect(secondary_13_4.n, load_13_4.p);
      connect(secondary_13_5.n, load_13_5.p);
      connect(secondary_13_6.n, load_13_6.p);
      connect(secondary_13_7.n, load_13_7.p);
      connect(secondary_13_8.n, load_13_8.p);
      connect(secondary_13_9.n, load_13_9.p);
      connect(secondary_13_10.n, load_13_10.p);
      connect(secondary_13_11.n, load_13_11.p);
      connect(secondary_13_12.n, load_13_12.p);
      connect(secondary_13_13.n, load_13_13.p);
      connect(secondary_13_14.n, load_13_14.p);
      connect(secondary_13_15.n, load_13_15.p);
      connect(secondary_13_16.n, load_13_16.p);
      connect(secondary_13_17.n, load_13_17.p);
      connect(secondary_13_18.n, load_13_18.p);
      connect(secondary_13_19.n, load_13_19.p);
      connect(secondary_13_20.n, load_13_20.p);
      connect(secondary_13_21.n, load_13_21.p);
      connect(secondary_13_22.n, load_13_22.p);
      connect(secondary_13_23.n, load_13_23.p);
      connect(secondary_13_24.n, load_13_24.p);
      connect(secondary_13_25.n, load_13_25.p);
      connect(secondary_13_26.n, load_13_26.p);
      connect(secondary_13_27.n, load_13_27.p);
      connect(secondary_13_28.n, load_13_28.p);
      connect(primary_14.n, secondary_14_1.p);
      connect(secondary_14_1.n, secondary_14_2.p);
      connect(secondary_14_2.n, secondary_14_3.p);
      connect(secondary_14_3.n, secondary_14_4.p);
      connect(secondary_14_4.n, secondary_14_5.p);
      connect(secondary_14_5.n, secondary_14_6.p);
      connect(secondary_14_6.n, secondary_14_7.p);
      connect(secondary_14_7.n, secondary_14_8.p);
      connect(secondary_14_8.n, secondary_14_9.p);
      connect(secondary_14_9.n, secondary_14_10.p);
      connect(secondary_14_10.n, secondary_14_11.p);
      connect(secondary_14_11.n, secondary_14_12.p);
      connect(secondary_14_12.n, secondary_14_13.p);
      connect(secondary_14_13.n, secondary_14_14.p);
      connect(secondary_14_14.n, secondary_14_15.p);
      connect(secondary_14_15.n, secondary_14_16.p);
      connect(secondary_14_16.n, secondary_14_17.p);
      connect(secondary_14_17.n, secondary_14_18.p);
      connect(secondary_14_18.n, secondary_14_19.p);
      connect(secondary_14_19.n, secondary_14_20.p);
      connect(secondary_14_20.n, secondary_14_21.p);
      connect(secondary_14_21.n, secondary_14_22.p);
      connect(secondary_14_22.n, secondary_14_23.p);
      connect(secondary_14_23.n, secondary_14_24.p);
      connect(secondary_14_24.n, secondary_14_25.p);
      connect(secondary_14_25.n, secondary_14_26.p);
      connect(secondary_14_26.n, secondary_14_27.p);
      connect(secondary_14_27.n, secondary_14_28.p);
      connect(secondary_14_1.n, load_14_1.p);
      connect(secondary_14_2.n, load_14_2.p);
      connect(secondary_14_3.n, load_14_3.p);
      connect(secondary_14_4.n, load_14_4.p);
      connect(secondary_14_5.n, load_14_5.p);
      connect(secondary_14_6.n, load_14_6.p);
      connect(secondary_14_7.n, load_14_7.p);
      connect(secondary_14_8.n, load_14_8.p);
      connect(secondary_14_9.n, load_14_9.p);
      connect(secondary_14_10.n, load_14_10.p);
      connect(secondary_14_11.n, load_14_11.p);
      connect(secondary_14_12.n, load_14_12.p);
      connect(secondary_14_13.n, load_14_13.p);
      connect(secondary_14_14.n, load_14_14.p);
      connect(secondary_14_15.n, load_14_15.p);
      connect(secondary_14_16.n, load_14_16.p);
      connect(secondary_14_17.n, load_14_17.p);
      connect(secondary_14_18.n, load_14_18.p);
      connect(secondary_14_19.n, load_14_19.p);
      connect(secondary_14_20.n, load_14_20.p);
      connect(secondary_14_21.n, load_14_21.p);
      connect(secondary_14_22.n, load_14_22.p);
      connect(secondary_14_23.n, load_14_23.p);
      connect(secondary_14_24.n, load_14_24.p);
      connect(secondary_14_25.n, load_14_25.p);
      connect(secondary_14_26.n, load_14_26.p);
      connect(secondary_14_27.n, load_14_27.p);
      connect(secondary_14_28.n, load_14_28.p);
      connect(primary_15.n, secondary_15_1.p);
      connect(secondary_15_1.n, secondary_15_2.p);
      connect(secondary_15_2.n, secondary_15_3.p);
      connect(secondary_15_3.n, secondary_15_4.p);
      connect(secondary_15_4.n, secondary_15_5.p);
      connect(secondary_15_5.n, secondary_15_6.p);
      connect(secondary_15_6.n, secondary_15_7.p);
      connect(secondary_15_7.n, secondary_15_8.p);
      connect(secondary_15_8.n, secondary_15_9.p);
      connect(secondary_15_9.n, secondary_15_10.p);
      connect(secondary_15_10.n, secondary_15_11.p);
      connect(secondary_15_11.n, secondary_15_12.p);
      connect(secondary_15_12.n, secondary_15_13.p);
      connect(secondary_15_13.n, secondary_15_14.p);
      connect(secondary_15_14.n, secondary_15_15.p);
      connect(secondary_15_15.n, secondary_15_16.p);
      connect(secondary_15_16.n, secondary_15_17.p);
      connect(secondary_15_17.n, secondary_15_18.p);
      connect(secondary_15_18.n, secondary_15_19.p);
      connect(secondary_15_19.n, secondary_15_20.p);
      connect(secondary_15_20.n, secondary_15_21.p);
      connect(secondary_15_21.n, secondary_15_22.p);
      connect(secondary_15_22.n, secondary_15_23.p);
      connect(secondary_15_23.n, secondary_15_24.p);
      connect(secondary_15_24.n, secondary_15_25.p);
      connect(secondary_15_25.n, secondary_15_26.p);
      connect(secondary_15_26.n, secondary_15_27.p);
      connect(secondary_15_27.n, secondary_15_28.p);
      connect(secondary_15_1.n, load_15_1.p);
      connect(secondary_15_2.n, load_15_2.p);
      connect(secondary_15_3.n, load_15_3.p);
      connect(secondary_15_4.n, load_15_4.p);
      connect(secondary_15_5.n, load_15_5.p);
      connect(secondary_15_6.n, load_15_6.p);
      connect(secondary_15_7.n, load_15_7.p);
      connect(secondary_15_8.n, load_15_8.p);
      connect(secondary_15_9.n, load_15_9.p);
      connect(secondary_15_10.n, load_15_10.p);
      connect(secondary_15_11.n, load_15_11.p);
      connect(secondary_15_12.n, load_15_12.p);
      connect(secondary_15_13.n, load_15_13.p);
      connect(secondary_15_14.n, load_15_14.p);
      connect(secondary_15_15.n, load_15_15.p);
      connect(secondary_15_16.n, load_15_16.p);
      connect(secondary_15_17.n, load_15_17.p);
      connect(secondary_15_18.n, load_15_18.p);
      connect(secondary_15_19.n, load_15_19.p);
      connect(secondary_15_20.n, load_15_20.p);
      connect(secondary_15_21.n, load_15_21.p);
      connect(secondary_15_22.n, load_15_22.p);
      connect(secondary_15_23.n, load_15_23.p);
      connect(secondary_15_24.n, load_15_24.p);
      connect(secondary_15_25.n, load_15_25.p);
      connect(secondary_15_26.n, load_15_26.p);
      connect(secondary_15_27.n, load_15_27.p);
      connect(secondary_15_28.n, load_15_28.p);
      connect(primary_16.n, secondary_16_1.p);
      connect(secondary_16_1.n, secondary_16_2.p);
      connect(secondary_16_2.n, secondary_16_3.p);
      connect(secondary_16_3.n, secondary_16_4.p);
      connect(secondary_16_4.n, secondary_16_5.p);
      connect(secondary_16_5.n, secondary_16_6.p);
      connect(secondary_16_6.n, secondary_16_7.p);
      connect(secondary_16_7.n, secondary_16_8.p);
      connect(secondary_16_8.n, secondary_16_9.p);
      connect(secondary_16_9.n, secondary_16_10.p);
      connect(secondary_16_10.n, secondary_16_11.p);
      connect(secondary_16_11.n, secondary_16_12.p);
      connect(secondary_16_12.n, secondary_16_13.p);
      connect(secondary_16_13.n, secondary_16_14.p);
      connect(secondary_16_14.n, secondary_16_15.p);
      connect(secondary_16_15.n, secondary_16_16.p);
      connect(secondary_16_16.n, secondary_16_17.p);
      connect(secondary_16_17.n, secondary_16_18.p);
      connect(secondary_16_18.n, secondary_16_19.p);
      connect(secondary_16_19.n, secondary_16_20.p);
      connect(secondary_16_20.n, secondary_16_21.p);
      connect(secondary_16_21.n, secondary_16_22.p);
      connect(secondary_16_22.n, secondary_16_23.p);
      connect(secondary_16_23.n, secondary_16_24.p);
      connect(secondary_16_24.n, secondary_16_25.p);
      connect(secondary_16_25.n, secondary_16_26.p);
      connect(secondary_16_26.n, secondary_16_27.p);
      connect(secondary_16_27.n, secondary_16_28.p);
      connect(secondary_16_1.n, load_16_1.p);
      connect(secondary_16_2.n, load_16_2.p);
      connect(secondary_16_3.n, load_16_3.p);
      connect(secondary_16_4.n, load_16_4.p);
      connect(secondary_16_5.n, load_16_5.p);
      connect(secondary_16_6.n, load_16_6.p);
      connect(secondary_16_7.n, load_16_7.p);
      connect(secondary_16_8.n, load_16_8.p);
      connect(secondary_16_9.n, load_16_9.p);
      connect(secondary_16_10.n, load_16_10.p);
      connect(secondary_16_11.n, load_16_11.p);
      connect(secondary_16_12.n, load_16_12.p);
      connect(secondary_16_13.n, load_16_13.p);
      connect(secondary_16_14.n, load_16_14.p);
      connect(secondary_16_15.n, load_16_15.p);
      connect(secondary_16_16.n, load_16_16.p);
      connect(secondary_16_17.n, load_16_17.p);
      connect(secondary_16_18.n, load_16_18.p);
      connect(secondary_16_19.n, load_16_19.p);
      connect(secondary_16_20.n, load_16_20.p);
      connect(secondary_16_21.n, load_16_21.p);
      connect(secondary_16_22.n, load_16_22.p);
      connect(secondary_16_23.n, load_16_23.p);
      connect(secondary_16_24.n, load_16_24.p);
      connect(secondary_16_25.n, load_16_25.p);
      connect(secondary_16_26.n, load_16_26.p);
      connect(secondary_16_27.n, load_16_27.p);
      connect(secondary_16_28.n, load_16_28.p);
      connect(primary_17.n, secondary_17_1.p);
      connect(secondary_17_1.n, secondary_17_2.p);
      connect(secondary_17_2.n, secondary_17_3.p);
      connect(secondary_17_3.n, secondary_17_4.p);
      connect(secondary_17_4.n, secondary_17_5.p);
      connect(secondary_17_5.n, secondary_17_6.p);
      connect(secondary_17_6.n, secondary_17_7.p);
      connect(secondary_17_7.n, secondary_17_8.p);
      connect(secondary_17_8.n, secondary_17_9.p);
      connect(secondary_17_9.n, secondary_17_10.p);
      connect(secondary_17_10.n, secondary_17_11.p);
      connect(secondary_17_11.n, secondary_17_12.p);
      connect(secondary_17_12.n, secondary_17_13.p);
      connect(secondary_17_13.n, secondary_17_14.p);
      connect(secondary_17_14.n, secondary_17_15.p);
      connect(secondary_17_15.n, secondary_17_16.p);
      connect(secondary_17_16.n, secondary_17_17.p);
      connect(secondary_17_17.n, secondary_17_18.p);
      connect(secondary_17_18.n, secondary_17_19.p);
      connect(secondary_17_19.n, secondary_17_20.p);
      connect(secondary_17_20.n, secondary_17_21.p);
      connect(secondary_17_21.n, secondary_17_22.p);
      connect(secondary_17_22.n, secondary_17_23.p);
      connect(secondary_17_23.n, secondary_17_24.p);
      connect(secondary_17_24.n, secondary_17_25.p);
      connect(secondary_17_25.n, secondary_17_26.p);
      connect(secondary_17_26.n, secondary_17_27.p);
      connect(secondary_17_27.n, secondary_17_28.p);
      connect(secondary_17_1.n, load_17_1.p);
      connect(secondary_17_2.n, load_17_2.p);
      connect(secondary_17_3.n, load_17_3.p);
      connect(secondary_17_4.n, load_17_4.p);
      connect(secondary_17_5.n, load_17_5.p);
      connect(secondary_17_6.n, load_17_6.p);
      connect(secondary_17_7.n, load_17_7.p);
      connect(secondary_17_8.n, load_17_8.p);
      connect(secondary_17_9.n, load_17_9.p);
      connect(secondary_17_10.n, load_17_10.p);
      connect(secondary_17_11.n, load_17_11.p);
      connect(secondary_17_12.n, load_17_12.p);
      connect(secondary_17_13.n, load_17_13.p);
      connect(secondary_17_14.n, load_17_14.p);
      connect(secondary_17_15.n, load_17_15.p);
      connect(secondary_17_16.n, load_17_16.p);
      connect(secondary_17_17.n, load_17_17.p);
      connect(secondary_17_18.n, load_17_18.p);
      connect(secondary_17_19.n, load_17_19.p);
      connect(secondary_17_20.n, load_17_20.p);
      connect(secondary_17_21.n, load_17_21.p);
      connect(secondary_17_22.n, load_17_22.p);
      connect(secondary_17_23.n, load_17_23.p);
      connect(secondary_17_24.n, load_17_24.p);
      connect(secondary_17_25.n, load_17_25.p);
      connect(secondary_17_26.n, load_17_26.p);
      connect(secondary_17_27.n, load_17_27.p);
      connect(secondary_17_28.n, load_17_28.p);
      connect(primary_18.n, secondary_18_1.p);
      connect(secondary_18_1.n, secondary_18_2.p);
      connect(secondary_18_2.n, secondary_18_3.p);
      connect(secondary_18_3.n, secondary_18_4.p);
      connect(secondary_18_4.n, secondary_18_5.p);
      connect(secondary_18_5.n, secondary_18_6.p);
      connect(secondary_18_6.n, secondary_18_7.p);
      connect(secondary_18_7.n, secondary_18_8.p);
      connect(secondary_18_8.n, secondary_18_9.p);
      connect(secondary_18_9.n, secondary_18_10.p);
      connect(secondary_18_10.n, secondary_18_11.p);
      connect(secondary_18_11.n, secondary_18_12.p);
      connect(secondary_18_12.n, secondary_18_13.p);
      connect(secondary_18_13.n, secondary_18_14.p);
      connect(secondary_18_14.n, secondary_18_15.p);
      connect(secondary_18_15.n, secondary_18_16.p);
      connect(secondary_18_16.n, secondary_18_17.p);
      connect(secondary_18_17.n, secondary_18_18.p);
      connect(secondary_18_18.n, secondary_18_19.p);
      connect(secondary_18_19.n, secondary_18_20.p);
      connect(secondary_18_20.n, secondary_18_21.p);
      connect(secondary_18_21.n, secondary_18_22.p);
      connect(secondary_18_22.n, secondary_18_23.p);
      connect(secondary_18_23.n, secondary_18_24.p);
      connect(secondary_18_24.n, secondary_18_25.p);
      connect(secondary_18_25.n, secondary_18_26.p);
      connect(secondary_18_26.n, secondary_18_27.p);
      connect(secondary_18_27.n, secondary_18_28.p);
      connect(secondary_18_1.n, load_18_1.p);
      connect(secondary_18_2.n, load_18_2.p);
      connect(secondary_18_3.n, load_18_3.p);
      connect(secondary_18_4.n, load_18_4.p);
      connect(secondary_18_5.n, load_18_5.p);
      connect(secondary_18_6.n, load_18_6.p);
      connect(secondary_18_7.n, load_18_7.p);
      connect(secondary_18_8.n, load_18_8.p);
      connect(secondary_18_9.n, load_18_9.p);
      connect(secondary_18_10.n, load_18_10.p);
      connect(secondary_18_11.n, load_18_11.p);
      connect(secondary_18_12.n, load_18_12.p);
      connect(secondary_18_13.n, load_18_13.p);
      connect(secondary_18_14.n, load_18_14.p);
      connect(secondary_18_15.n, load_18_15.p);
      connect(secondary_18_16.n, load_18_16.p);
      connect(secondary_18_17.n, load_18_17.p);
      connect(secondary_18_18.n, load_18_18.p);
      connect(secondary_18_19.n, load_18_19.p);
      connect(secondary_18_20.n, load_18_20.p);
      connect(secondary_18_21.n, load_18_21.p);
      connect(secondary_18_22.n, load_18_22.p);
      connect(secondary_18_23.n, load_18_23.p);
      connect(secondary_18_24.n, load_18_24.p);
      connect(secondary_18_25.n, load_18_25.p);
      connect(secondary_18_26.n, load_18_26.p);
      connect(secondary_18_27.n, load_18_27.p);
      connect(secondary_18_28.n, load_18_28.p);
      connect(primary_19.n, secondary_19_1.p);
      connect(secondary_19_1.n, secondary_19_2.p);
      connect(secondary_19_2.n, secondary_19_3.p);
      connect(secondary_19_3.n, secondary_19_4.p);
      connect(secondary_19_4.n, secondary_19_5.p);
      connect(secondary_19_5.n, secondary_19_6.p);
      connect(secondary_19_6.n, secondary_19_7.p);
      connect(secondary_19_7.n, secondary_19_8.p);
      connect(secondary_19_8.n, secondary_19_9.p);
      connect(secondary_19_9.n, secondary_19_10.p);
      connect(secondary_19_10.n, secondary_19_11.p);
      connect(secondary_19_11.n, secondary_19_12.p);
      connect(secondary_19_12.n, secondary_19_13.p);
      connect(secondary_19_13.n, secondary_19_14.p);
      connect(secondary_19_14.n, secondary_19_15.p);
      connect(secondary_19_15.n, secondary_19_16.p);
      connect(secondary_19_16.n, secondary_19_17.p);
      connect(secondary_19_17.n, secondary_19_18.p);
      connect(secondary_19_18.n, secondary_19_19.p);
      connect(secondary_19_19.n, secondary_19_20.p);
      connect(secondary_19_20.n, secondary_19_21.p);
      connect(secondary_19_21.n, secondary_19_22.p);
      connect(secondary_19_22.n, secondary_19_23.p);
      connect(secondary_19_23.n, secondary_19_24.p);
      connect(secondary_19_24.n, secondary_19_25.p);
      connect(secondary_19_25.n, secondary_19_26.p);
      connect(secondary_19_26.n, secondary_19_27.p);
      connect(secondary_19_27.n, secondary_19_28.p);
      connect(secondary_19_1.n, load_19_1.p);
      connect(secondary_19_2.n, load_19_2.p);
      connect(secondary_19_3.n, load_19_3.p);
      connect(secondary_19_4.n, load_19_4.p);
      connect(secondary_19_5.n, load_19_5.p);
      connect(secondary_19_6.n, load_19_6.p);
      connect(secondary_19_7.n, load_19_7.p);
      connect(secondary_19_8.n, load_19_8.p);
      connect(secondary_19_9.n, load_19_9.p);
      connect(secondary_19_10.n, load_19_10.p);
      connect(secondary_19_11.n, load_19_11.p);
      connect(secondary_19_12.n, load_19_12.p);
      connect(secondary_19_13.n, load_19_13.p);
      connect(secondary_19_14.n, load_19_14.p);
      connect(secondary_19_15.n, load_19_15.p);
      connect(secondary_19_16.n, load_19_16.p);
      connect(secondary_19_17.n, load_19_17.p);
      connect(secondary_19_18.n, load_19_18.p);
      connect(secondary_19_19.n, load_19_19.p);
      connect(secondary_19_20.n, load_19_20.p);
      connect(secondary_19_21.n, load_19_21.p);
      connect(secondary_19_22.n, load_19_22.p);
      connect(secondary_19_23.n, load_19_23.p);
      connect(secondary_19_24.n, load_19_24.p);
      connect(secondary_19_25.n, load_19_25.p);
      connect(secondary_19_26.n, load_19_26.p);
      connect(secondary_19_27.n, load_19_27.p);
      connect(secondary_19_28.n, load_19_28.p);
      connect(primary_20.n, secondary_20_1.p);
      connect(secondary_20_1.n, secondary_20_2.p);
      connect(secondary_20_2.n, secondary_20_3.p);
      connect(secondary_20_3.n, secondary_20_4.p);
      connect(secondary_20_4.n, secondary_20_5.p);
      connect(secondary_20_5.n, secondary_20_6.p);
      connect(secondary_20_6.n, secondary_20_7.p);
      connect(secondary_20_7.n, secondary_20_8.p);
      connect(secondary_20_8.n, secondary_20_9.p);
      connect(secondary_20_9.n, secondary_20_10.p);
      connect(secondary_20_10.n, secondary_20_11.p);
      connect(secondary_20_11.n, secondary_20_12.p);
      connect(secondary_20_12.n, secondary_20_13.p);
      connect(secondary_20_13.n, secondary_20_14.p);
      connect(secondary_20_14.n, secondary_20_15.p);
      connect(secondary_20_15.n, secondary_20_16.p);
      connect(secondary_20_16.n, secondary_20_17.p);
      connect(secondary_20_17.n, secondary_20_18.p);
      connect(secondary_20_18.n, secondary_20_19.p);
      connect(secondary_20_19.n, secondary_20_20.p);
      connect(secondary_20_20.n, secondary_20_21.p);
      connect(secondary_20_21.n, secondary_20_22.p);
      connect(secondary_20_22.n, secondary_20_23.p);
      connect(secondary_20_23.n, secondary_20_24.p);
      connect(secondary_20_24.n, secondary_20_25.p);
      connect(secondary_20_25.n, secondary_20_26.p);
      connect(secondary_20_26.n, secondary_20_27.p);
      connect(secondary_20_27.n, secondary_20_28.p);
      connect(secondary_20_1.n, load_20_1.p);
      connect(secondary_20_2.n, load_20_2.p);
      connect(secondary_20_3.n, load_20_3.p);
      connect(secondary_20_4.n, load_20_4.p);
      connect(secondary_20_5.n, load_20_5.p);
      connect(secondary_20_6.n, load_20_6.p);
      connect(secondary_20_7.n, load_20_7.p);
      connect(secondary_20_8.n, load_20_8.p);
      connect(secondary_20_9.n, load_20_9.p);
      connect(secondary_20_10.n, load_20_10.p);
      connect(secondary_20_11.n, load_20_11.p);
      connect(secondary_20_12.n, load_20_12.p);
      connect(secondary_20_13.n, load_20_13.p);
      connect(secondary_20_14.n, load_20_14.p);
      connect(secondary_20_15.n, load_20_15.p);
      connect(secondary_20_16.n, load_20_16.p);
      connect(secondary_20_17.n, load_20_17.p);
      connect(secondary_20_18.n, load_20_18.p);
      connect(secondary_20_19.n, load_20_19.p);
      connect(secondary_20_20.n, load_20_20.p);
      connect(secondary_20_21.n, load_20_21.p);
      connect(secondary_20_22.n, load_20_22.p);
      connect(secondary_20_23.n, load_20_23.p);
      connect(secondary_20_24.n, load_20_24.p);
      connect(secondary_20_25.n, load_20_25.p);
      connect(secondary_20_26.n, load_20_26.p);
      connect(secondary_20_27.n, load_20_27.p);
      connect(secondary_20_28.n, load_20_28.p);
      connect(primary_21.n, secondary_21_1.p);
      connect(secondary_21_1.n, secondary_21_2.p);
      connect(secondary_21_2.n, secondary_21_3.p);
      connect(secondary_21_3.n, secondary_21_4.p);
      connect(secondary_21_4.n, secondary_21_5.p);
      connect(secondary_21_5.n, secondary_21_6.p);
      connect(secondary_21_6.n, secondary_21_7.p);
      connect(secondary_21_7.n, secondary_21_8.p);
      connect(secondary_21_8.n, secondary_21_9.p);
      connect(secondary_21_9.n, secondary_21_10.p);
      connect(secondary_21_10.n, secondary_21_11.p);
      connect(secondary_21_11.n, secondary_21_12.p);
      connect(secondary_21_12.n, secondary_21_13.p);
      connect(secondary_21_13.n, secondary_21_14.p);
      connect(secondary_21_14.n, secondary_21_15.p);
      connect(secondary_21_15.n, secondary_21_16.p);
      connect(secondary_21_16.n, secondary_21_17.p);
      connect(secondary_21_17.n, secondary_21_18.p);
      connect(secondary_21_18.n, secondary_21_19.p);
      connect(secondary_21_19.n, secondary_21_20.p);
      connect(secondary_21_20.n, secondary_21_21.p);
      connect(secondary_21_21.n, secondary_21_22.p);
      connect(secondary_21_22.n, secondary_21_23.p);
      connect(secondary_21_23.n, secondary_21_24.p);
      connect(secondary_21_24.n, secondary_21_25.p);
      connect(secondary_21_25.n, secondary_21_26.p);
      connect(secondary_21_26.n, secondary_21_27.p);
      connect(secondary_21_27.n, secondary_21_28.p);
      connect(secondary_21_1.n, load_21_1.p);
      connect(secondary_21_2.n, load_21_2.p);
      connect(secondary_21_3.n, load_21_3.p);
      connect(secondary_21_4.n, load_21_4.p);
      connect(secondary_21_5.n, load_21_5.p);
      connect(secondary_21_6.n, load_21_6.p);
      connect(secondary_21_7.n, load_21_7.p);
      connect(secondary_21_8.n, load_21_8.p);
      connect(secondary_21_9.n, load_21_9.p);
      connect(secondary_21_10.n, load_21_10.p);
      connect(secondary_21_11.n, load_21_11.p);
      connect(secondary_21_12.n, load_21_12.p);
      connect(secondary_21_13.n, load_21_13.p);
      connect(secondary_21_14.n, load_21_14.p);
      connect(secondary_21_15.n, load_21_15.p);
      connect(secondary_21_16.n, load_21_16.p);
      connect(secondary_21_17.n, load_21_17.p);
      connect(secondary_21_18.n, load_21_18.p);
      connect(secondary_21_19.n, load_21_19.p);
      connect(secondary_21_20.n, load_21_20.p);
      connect(secondary_21_21.n, load_21_21.p);
      connect(secondary_21_22.n, load_21_22.p);
      connect(secondary_21_23.n, load_21_23.p);
      connect(secondary_21_24.n, load_21_24.p);
      connect(secondary_21_25.n, load_21_25.p);
      connect(secondary_21_26.n, load_21_26.p);
      connect(secondary_21_27.n, load_21_27.p);
      connect(secondary_21_28.n, load_21_28.p);
      connect(primary_22.n, secondary_22_1.p);
      connect(secondary_22_1.n, secondary_22_2.p);
      connect(secondary_22_2.n, secondary_22_3.p);
      connect(secondary_22_3.n, secondary_22_4.p);
      connect(secondary_22_4.n, secondary_22_5.p);
      connect(secondary_22_5.n, secondary_22_6.p);
      connect(secondary_22_6.n, secondary_22_7.p);
      connect(secondary_22_7.n, secondary_22_8.p);
      connect(secondary_22_8.n, secondary_22_9.p);
      connect(secondary_22_9.n, secondary_22_10.p);
      connect(secondary_22_10.n, secondary_22_11.p);
      connect(secondary_22_11.n, secondary_22_12.p);
      connect(secondary_22_12.n, secondary_22_13.p);
      connect(secondary_22_13.n, secondary_22_14.p);
      connect(secondary_22_14.n, secondary_22_15.p);
      connect(secondary_22_15.n, secondary_22_16.p);
      connect(secondary_22_16.n, secondary_22_17.p);
      connect(secondary_22_17.n, secondary_22_18.p);
      connect(secondary_22_18.n, secondary_22_19.p);
      connect(secondary_22_19.n, secondary_22_20.p);
      connect(secondary_22_20.n, secondary_22_21.p);
      connect(secondary_22_21.n, secondary_22_22.p);
      connect(secondary_22_22.n, secondary_22_23.p);
      connect(secondary_22_23.n, secondary_22_24.p);
      connect(secondary_22_24.n, secondary_22_25.p);
      connect(secondary_22_25.n, secondary_22_26.p);
      connect(secondary_22_26.n, secondary_22_27.p);
      connect(secondary_22_27.n, secondary_22_28.p);
      connect(secondary_22_1.n, load_22_1.p);
      connect(secondary_22_2.n, load_22_2.p);
      connect(secondary_22_3.n, load_22_3.p);
      connect(secondary_22_4.n, load_22_4.p);
      connect(secondary_22_5.n, load_22_5.p);
      connect(secondary_22_6.n, load_22_6.p);
      connect(secondary_22_7.n, load_22_7.p);
      connect(secondary_22_8.n, load_22_8.p);
      connect(secondary_22_9.n, load_22_9.p);
      connect(secondary_22_10.n, load_22_10.p);
      connect(secondary_22_11.n, load_22_11.p);
      connect(secondary_22_12.n, load_22_12.p);
      connect(secondary_22_13.n, load_22_13.p);
      connect(secondary_22_14.n, load_22_14.p);
      connect(secondary_22_15.n, load_22_15.p);
      connect(secondary_22_16.n, load_22_16.p);
      connect(secondary_22_17.n, load_22_17.p);
      connect(secondary_22_18.n, load_22_18.p);
      connect(secondary_22_19.n, load_22_19.p);
      connect(secondary_22_20.n, load_22_20.p);
      connect(secondary_22_21.n, load_22_21.p);
      connect(secondary_22_22.n, load_22_22.p);
      connect(secondary_22_23.n, load_22_23.p);
      connect(secondary_22_24.n, load_22_24.p);
      connect(secondary_22_25.n, load_22_25.p);
      connect(secondary_22_26.n, load_22_26.p);
      connect(secondary_22_27.n, load_22_27.p);
      connect(secondary_22_28.n, load_22_28.p);
      connect(primary_23.n, secondary_23_1.p);
      connect(secondary_23_1.n, secondary_23_2.p);
      connect(secondary_23_2.n, secondary_23_3.p);
      connect(secondary_23_3.n, secondary_23_4.p);
      connect(secondary_23_4.n, secondary_23_5.p);
      connect(secondary_23_5.n, secondary_23_6.p);
      connect(secondary_23_6.n, secondary_23_7.p);
      connect(secondary_23_7.n, secondary_23_8.p);
      connect(secondary_23_8.n, secondary_23_9.p);
      connect(secondary_23_9.n, secondary_23_10.p);
      connect(secondary_23_10.n, secondary_23_11.p);
      connect(secondary_23_11.n, secondary_23_12.p);
      connect(secondary_23_12.n, secondary_23_13.p);
      connect(secondary_23_13.n, secondary_23_14.p);
      connect(secondary_23_14.n, secondary_23_15.p);
      connect(secondary_23_15.n, secondary_23_16.p);
      connect(secondary_23_16.n, secondary_23_17.p);
      connect(secondary_23_17.n, secondary_23_18.p);
      connect(secondary_23_18.n, secondary_23_19.p);
      connect(secondary_23_19.n, secondary_23_20.p);
      connect(secondary_23_20.n, secondary_23_21.p);
      connect(secondary_23_21.n, secondary_23_22.p);
      connect(secondary_23_22.n, secondary_23_23.p);
      connect(secondary_23_23.n, secondary_23_24.p);
      connect(secondary_23_24.n, secondary_23_25.p);
      connect(secondary_23_25.n, secondary_23_26.p);
      connect(secondary_23_26.n, secondary_23_27.p);
      connect(secondary_23_27.n, secondary_23_28.p);
      connect(secondary_23_1.n, load_23_1.p);
      connect(secondary_23_2.n, load_23_2.p);
      connect(secondary_23_3.n, load_23_3.p);
      connect(secondary_23_4.n, load_23_4.p);
      connect(secondary_23_5.n, load_23_5.p);
      connect(secondary_23_6.n, load_23_6.p);
      connect(secondary_23_7.n, load_23_7.p);
      connect(secondary_23_8.n, load_23_8.p);
      connect(secondary_23_9.n, load_23_9.p);
      connect(secondary_23_10.n, load_23_10.p);
      connect(secondary_23_11.n, load_23_11.p);
      connect(secondary_23_12.n, load_23_12.p);
      connect(secondary_23_13.n, load_23_13.p);
      connect(secondary_23_14.n, load_23_14.p);
      connect(secondary_23_15.n, load_23_15.p);
      connect(secondary_23_16.n, load_23_16.p);
      connect(secondary_23_17.n, load_23_17.p);
      connect(secondary_23_18.n, load_23_18.p);
      connect(secondary_23_19.n, load_23_19.p);
      connect(secondary_23_20.n, load_23_20.p);
      connect(secondary_23_21.n, load_23_21.p);
      connect(secondary_23_22.n, load_23_22.p);
      connect(secondary_23_23.n, load_23_23.p);
      connect(secondary_23_24.n, load_23_24.p);
      connect(secondary_23_25.n, load_23_25.p);
      connect(secondary_23_26.n, load_23_26.p);
      connect(secondary_23_27.n, load_23_27.p);
      connect(secondary_23_28.n, load_23_28.p);
      connect(primary_24.n, secondary_24_1.p);
      connect(secondary_24_1.n, secondary_24_2.p);
      connect(secondary_24_2.n, secondary_24_3.p);
      connect(secondary_24_3.n, secondary_24_4.p);
      connect(secondary_24_4.n, secondary_24_5.p);
      connect(secondary_24_5.n, secondary_24_6.p);
      connect(secondary_24_6.n, secondary_24_7.p);
      connect(secondary_24_7.n, secondary_24_8.p);
      connect(secondary_24_8.n, secondary_24_9.p);
      connect(secondary_24_9.n, secondary_24_10.p);
      connect(secondary_24_10.n, secondary_24_11.p);
      connect(secondary_24_11.n, secondary_24_12.p);
      connect(secondary_24_12.n, secondary_24_13.p);
      connect(secondary_24_13.n, secondary_24_14.p);
      connect(secondary_24_14.n, secondary_24_15.p);
      connect(secondary_24_15.n, secondary_24_16.p);
      connect(secondary_24_16.n, secondary_24_17.p);
      connect(secondary_24_17.n, secondary_24_18.p);
      connect(secondary_24_18.n, secondary_24_19.p);
      connect(secondary_24_19.n, secondary_24_20.p);
      connect(secondary_24_20.n, secondary_24_21.p);
      connect(secondary_24_21.n, secondary_24_22.p);
      connect(secondary_24_22.n, secondary_24_23.p);
      connect(secondary_24_23.n, secondary_24_24.p);
      connect(secondary_24_24.n, secondary_24_25.p);
      connect(secondary_24_25.n, secondary_24_26.p);
      connect(secondary_24_26.n, secondary_24_27.p);
      connect(secondary_24_27.n, secondary_24_28.p);
      connect(secondary_24_1.n, load_24_1.p);
      connect(secondary_24_2.n, load_24_2.p);
      connect(secondary_24_3.n, load_24_3.p);
      connect(secondary_24_4.n, load_24_4.p);
      connect(secondary_24_5.n, load_24_5.p);
      connect(secondary_24_6.n, load_24_6.p);
      connect(secondary_24_7.n, load_24_7.p);
      connect(secondary_24_8.n, load_24_8.p);
      connect(secondary_24_9.n, load_24_9.p);
      connect(secondary_24_10.n, load_24_10.p);
      connect(secondary_24_11.n, load_24_11.p);
      connect(secondary_24_12.n, load_24_12.p);
      connect(secondary_24_13.n, load_24_13.p);
      connect(secondary_24_14.n, load_24_14.p);
      connect(secondary_24_15.n, load_24_15.p);
      connect(secondary_24_16.n, load_24_16.p);
      connect(secondary_24_17.n, load_24_17.p);
      connect(secondary_24_18.n, load_24_18.p);
      connect(secondary_24_19.n, load_24_19.p);
      connect(secondary_24_20.n, load_24_20.p);
      connect(secondary_24_21.n, load_24_21.p);
      connect(secondary_24_22.n, load_24_22.p);
      connect(secondary_24_23.n, load_24_23.p);
      connect(secondary_24_24.n, load_24_24.p);
      connect(secondary_24_25.n, load_24_25.p);
      connect(secondary_24_26.n, load_24_26.p);
      connect(secondary_24_27.n, load_24_27.p);
      connect(secondary_24_28.n, load_24_28.p);
      connect(primary_25.n, secondary_25_1.p);
      connect(secondary_25_1.n, secondary_25_2.p);
      connect(secondary_25_2.n, secondary_25_3.p);
      connect(secondary_25_3.n, secondary_25_4.p);
      connect(secondary_25_4.n, secondary_25_5.p);
      connect(secondary_25_5.n, secondary_25_6.p);
      connect(secondary_25_6.n, secondary_25_7.p);
      connect(secondary_25_7.n, secondary_25_8.p);
      connect(secondary_25_8.n, secondary_25_9.p);
      connect(secondary_25_9.n, secondary_25_10.p);
      connect(secondary_25_10.n, secondary_25_11.p);
      connect(secondary_25_11.n, secondary_25_12.p);
      connect(secondary_25_12.n, secondary_25_13.p);
      connect(secondary_25_13.n, secondary_25_14.p);
      connect(secondary_25_14.n, secondary_25_15.p);
      connect(secondary_25_15.n, secondary_25_16.p);
      connect(secondary_25_16.n, secondary_25_17.p);
      connect(secondary_25_17.n, secondary_25_18.p);
      connect(secondary_25_18.n, secondary_25_19.p);
      connect(secondary_25_19.n, secondary_25_20.p);
      connect(secondary_25_20.n, secondary_25_21.p);
      connect(secondary_25_21.n, secondary_25_22.p);
      connect(secondary_25_22.n, secondary_25_23.p);
      connect(secondary_25_23.n, secondary_25_24.p);
      connect(secondary_25_24.n, secondary_25_25.p);
      connect(secondary_25_25.n, secondary_25_26.p);
      connect(secondary_25_26.n, secondary_25_27.p);
      connect(secondary_25_27.n, secondary_25_28.p);
      connect(secondary_25_1.n, load_25_1.p);
      connect(secondary_25_2.n, load_25_2.p);
      connect(secondary_25_3.n, load_25_3.p);
      connect(secondary_25_4.n, load_25_4.p);
      connect(secondary_25_5.n, load_25_5.p);
      connect(secondary_25_6.n, load_25_6.p);
      connect(secondary_25_7.n, load_25_7.p);
      connect(secondary_25_8.n, load_25_8.p);
      connect(secondary_25_9.n, load_25_9.p);
      connect(secondary_25_10.n, load_25_10.p);
      connect(secondary_25_11.n, load_25_11.p);
      connect(secondary_25_12.n, load_25_12.p);
      connect(secondary_25_13.n, load_25_13.p);
      connect(secondary_25_14.n, load_25_14.p);
      connect(secondary_25_15.n, load_25_15.p);
      connect(secondary_25_16.n, load_25_16.p);
      connect(secondary_25_17.n, load_25_17.p);
      connect(secondary_25_18.n, load_25_18.p);
      connect(secondary_25_19.n, load_25_19.p);
      connect(secondary_25_20.n, load_25_20.p);
      connect(secondary_25_21.n, load_25_21.p);
      connect(secondary_25_22.n, load_25_22.p);
      connect(secondary_25_23.n, load_25_23.p);
      connect(secondary_25_24.n, load_25_24.p);
      connect(secondary_25_25.n, load_25_25.p);
      connect(secondary_25_26.n, load_25_26.p);
      connect(secondary_25_27.n, load_25_27.p);
      connect(secondary_25_28.n, load_25_28.p);
      connect(primary_26.n, secondary_26_1.p);
      connect(secondary_26_1.n, secondary_26_2.p);
      connect(secondary_26_2.n, secondary_26_3.p);
      connect(secondary_26_3.n, secondary_26_4.p);
      connect(secondary_26_4.n, secondary_26_5.p);
      connect(secondary_26_5.n, secondary_26_6.p);
      connect(secondary_26_6.n, secondary_26_7.p);
      connect(secondary_26_7.n, secondary_26_8.p);
      connect(secondary_26_8.n, secondary_26_9.p);
      connect(secondary_26_9.n, secondary_26_10.p);
      connect(secondary_26_10.n, secondary_26_11.p);
      connect(secondary_26_11.n, secondary_26_12.p);
      connect(secondary_26_12.n, secondary_26_13.p);
      connect(secondary_26_13.n, secondary_26_14.p);
      connect(secondary_26_14.n, secondary_26_15.p);
      connect(secondary_26_15.n, secondary_26_16.p);
      connect(secondary_26_16.n, secondary_26_17.p);
      connect(secondary_26_17.n, secondary_26_18.p);
      connect(secondary_26_18.n, secondary_26_19.p);
      connect(secondary_26_19.n, secondary_26_20.p);
      connect(secondary_26_20.n, secondary_26_21.p);
      connect(secondary_26_21.n, secondary_26_22.p);
      connect(secondary_26_22.n, secondary_26_23.p);
      connect(secondary_26_23.n, secondary_26_24.p);
      connect(secondary_26_24.n, secondary_26_25.p);
      connect(secondary_26_25.n, secondary_26_26.p);
      connect(secondary_26_26.n, secondary_26_27.p);
      connect(secondary_26_27.n, secondary_26_28.p);
      connect(secondary_26_1.n, load_26_1.p);
      connect(secondary_26_2.n, load_26_2.p);
      connect(secondary_26_3.n, load_26_3.p);
      connect(secondary_26_4.n, load_26_4.p);
      connect(secondary_26_5.n, load_26_5.p);
      connect(secondary_26_6.n, load_26_6.p);
      connect(secondary_26_7.n, load_26_7.p);
      connect(secondary_26_8.n, load_26_8.p);
      connect(secondary_26_9.n, load_26_9.p);
      connect(secondary_26_10.n, load_26_10.p);
      connect(secondary_26_11.n, load_26_11.p);
      connect(secondary_26_12.n, load_26_12.p);
      connect(secondary_26_13.n, load_26_13.p);
      connect(secondary_26_14.n, load_26_14.p);
      connect(secondary_26_15.n, load_26_15.p);
      connect(secondary_26_16.n, load_26_16.p);
      connect(secondary_26_17.n, load_26_17.p);
      connect(secondary_26_18.n, load_26_18.p);
      connect(secondary_26_19.n, load_26_19.p);
      connect(secondary_26_20.n, load_26_20.p);
      connect(secondary_26_21.n, load_26_21.p);
      connect(secondary_26_22.n, load_26_22.p);
      connect(secondary_26_23.n, load_26_23.p);
      connect(secondary_26_24.n, load_26_24.p);
      connect(secondary_26_25.n, load_26_25.p);
      connect(secondary_26_26.n, load_26_26.p);
      connect(secondary_26_27.n, load_26_27.p);
      connect(secondary_26_28.n, load_26_28.p);
      connect(primary_27.n, secondary_27_1.p);
      connect(secondary_27_1.n, secondary_27_2.p);
      connect(secondary_27_2.n, secondary_27_3.p);
      connect(secondary_27_3.n, secondary_27_4.p);
      connect(secondary_27_4.n, secondary_27_5.p);
      connect(secondary_27_5.n, secondary_27_6.p);
      connect(secondary_27_6.n, secondary_27_7.p);
      connect(secondary_27_7.n, secondary_27_8.p);
      connect(secondary_27_8.n, secondary_27_9.p);
      connect(secondary_27_9.n, secondary_27_10.p);
      connect(secondary_27_10.n, secondary_27_11.p);
      connect(secondary_27_11.n, secondary_27_12.p);
      connect(secondary_27_12.n, secondary_27_13.p);
      connect(secondary_27_13.n, secondary_27_14.p);
      connect(secondary_27_14.n, secondary_27_15.p);
      connect(secondary_27_15.n, secondary_27_16.p);
      connect(secondary_27_16.n, secondary_27_17.p);
      connect(secondary_27_17.n, secondary_27_18.p);
      connect(secondary_27_18.n, secondary_27_19.p);
      connect(secondary_27_19.n, secondary_27_20.p);
      connect(secondary_27_20.n, secondary_27_21.p);
      connect(secondary_27_21.n, secondary_27_22.p);
      connect(secondary_27_22.n, secondary_27_23.p);
      connect(secondary_27_23.n, secondary_27_24.p);
      connect(secondary_27_24.n, secondary_27_25.p);
      connect(secondary_27_25.n, secondary_27_26.p);
      connect(secondary_27_26.n, secondary_27_27.p);
      connect(secondary_27_27.n, secondary_27_28.p);
      connect(secondary_27_1.n, load_27_1.p);
      connect(secondary_27_2.n, load_27_2.p);
      connect(secondary_27_3.n, load_27_3.p);
      connect(secondary_27_4.n, load_27_4.p);
      connect(secondary_27_5.n, load_27_5.p);
      connect(secondary_27_6.n, load_27_6.p);
      connect(secondary_27_7.n, load_27_7.p);
      connect(secondary_27_8.n, load_27_8.p);
      connect(secondary_27_9.n, load_27_9.p);
      connect(secondary_27_10.n, load_27_10.p);
      connect(secondary_27_11.n, load_27_11.p);
      connect(secondary_27_12.n, load_27_12.p);
      connect(secondary_27_13.n, load_27_13.p);
      connect(secondary_27_14.n, load_27_14.p);
      connect(secondary_27_15.n, load_27_15.p);
      connect(secondary_27_16.n, load_27_16.p);
      connect(secondary_27_17.n, load_27_17.p);
      connect(secondary_27_18.n, load_27_18.p);
      connect(secondary_27_19.n, load_27_19.p);
      connect(secondary_27_20.n, load_27_20.p);
      connect(secondary_27_21.n, load_27_21.p);
      connect(secondary_27_22.n, load_27_22.p);
      connect(secondary_27_23.n, load_27_23.p);
      connect(secondary_27_24.n, load_27_24.p);
      connect(secondary_27_25.n, load_27_25.p);
      connect(secondary_27_26.n, load_27_26.p);
      connect(secondary_27_27.n, load_27_27.p);
      connect(secondary_27_28.n, load_27_28.p);
      connect(primary_28.n, secondary_28_1.p);
      connect(secondary_28_1.n, secondary_28_2.p);
      connect(secondary_28_2.n, secondary_28_3.p);
      connect(secondary_28_3.n, secondary_28_4.p);
      connect(secondary_28_4.n, secondary_28_5.p);
      connect(secondary_28_5.n, secondary_28_6.p);
      connect(secondary_28_6.n, secondary_28_7.p);
      connect(secondary_28_7.n, secondary_28_8.p);
      connect(secondary_28_8.n, secondary_28_9.p);
      connect(secondary_28_9.n, secondary_28_10.p);
      connect(secondary_28_10.n, secondary_28_11.p);
      connect(secondary_28_11.n, secondary_28_12.p);
      connect(secondary_28_12.n, secondary_28_13.p);
      connect(secondary_28_13.n, secondary_28_14.p);
      connect(secondary_28_14.n, secondary_28_15.p);
      connect(secondary_28_15.n, secondary_28_16.p);
      connect(secondary_28_16.n, secondary_28_17.p);
      connect(secondary_28_17.n, secondary_28_18.p);
      connect(secondary_28_18.n, secondary_28_19.p);
      connect(secondary_28_19.n, secondary_28_20.p);
      connect(secondary_28_20.n, secondary_28_21.p);
      connect(secondary_28_21.n, secondary_28_22.p);
      connect(secondary_28_22.n, secondary_28_23.p);
      connect(secondary_28_23.n, secondary_28_24.p);
      connect(secondary_28_24.n, secondary_28_25.p);
      connect(secondary_28_25.n, secondary_28_26.p);
      connect(secondary_28_26.n, secondary_28_27.p);
      connect(secondary_28_27.n, secondary_28_28.p);
      connect(secondary_28_1.n, load_28_1.p);
      connect(secondary_28_2.n, load_28_2.p);
      connect(secondary_28_3.n, load_28_3.p);
      connect(secondary_28_4.n, load_28_4.p);
      connect(secondary_28_5.n, load_28_5.p);
      connect(secondary_28_6.n, load_28_6.p);
      connect(secondary_28_7.n, load_28_7.p);
      connect(secondary_28_8.n, load_28_8.p);
      connect(secondary_28_9.n, load_28_9.p);
      connect(secondary_28_10.n, load_28_10.p);
      connect(secondary_28_11.n, load_28_11.p);
      connect(secondary_28_12.n, load_28_12.p);
      connect(secondary_28_13.n, load_28_13.p);
      connect(secondary_28_14.n, load_28_14.p);
      connect(secondary_28_15.n, load_28_15.p);
      connect(secondary_28_16.n, load_28_16.p);
      connect(secondary_28_17.n, load_28_17.p);
      connect(secondary_28_18.n, load_28_18.p);
      connect(secondary_28_19.n, load_28_19.p);
      connect(secondary_28_20.n, load_28_20.p);
      connect(secondary_28_21.n, load_28_21.p);
      connect(secondary_28_22.n, load_28_22.p);
      connect(secondary_28_23.n, load_28_23.p);
      connect(secondary_28_24.n, load_28_24.p);
      connect(secondary_28_25.n, load_28_25.p);
      connect(secondary_28_26.n, load_28_26.p);
      connect(secondary_28_27.n, load_28_27.p);
      connect(secondary_28_28.n, load_28_28.p);
      annotation(experiment(StopTime = 1, Interval = 1e-3));
    end DistributionSystemLinearIndividual_N_28_M_28;
  end ScaledExperiments;
end DistributionSystemAC;
